"""The GroupSetDialog dialog."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules
from PySide2.QtCore import QPoint, Qt
from PySide2.QtGui import QIcon, QKeySequence, QPalette
from PySide2.QtWidgets import QDialogButtonBox, QInputDialog, QListWidgetItem, QMessageBox, QShortcut, QToolTip, \
    QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.guipy.resources.resources_util import get_resource_path
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.gmi.data.component_data import CurveAdder, CurveGetter
from xms.gmi.data.generic_model import GroupSet
from xms.gmi.gui.base_dlg import BaseDialog
from xms.gmi.gui.dataset_callback import DatasetCallback
from xms.gmi.gui.group_set_dialog_ui import Ui_GroupSetDialog
from xms.gmi.gui.parameter_widgets import make_widgets, ParameterWidget


class QxListWidgetItem(QListWidgetItem):
    """A QListWidgetItem with an extra group_id member."""
    def __init__(self, group_id, *args, **kwargs):
        """
        Initialize the widget.

        Args:
            group_id: ID of the group this widget is in.
            *args: Passed through to QListWidgetItem constructor.
            **kwargs: Passed through to QListWidgetItem constructor.
        """
        self.group_id = group_id
        super().__init__(*args, **kwargs)
        self.setFlags(self.flags() | Qt.ItemIsEditable)


class SectionDialog(Ui_GroupSetDialog, BaseDialog):
    """
    A dialog for editing model parameters.

    This dialog accepts an `xms.gmi.data.generic_model.Section` and creates widgets for editing the parameters in it.
    If values are provided in the section, they will be shown in the dialog. When the dialog finishes, the values in
    the section will be updated to reflect the user's choices.
    """
    def __init__(
        self,
        parent: Optional[QWidget],
        section: GroupSet,
        get_curve: Optional[CurveGetter] = None,
        add_curve: Optional[CurveAdder] = None,
        is_interior: bool = False,
        dlg_name: str = '',
        window_title: str = '',
        multi_select_message: str = '',
        banner: Optional[tuple[str, str]] = None,
        show_groups: bool = True,
        enable_unchecked_groups: bool = False,
        hide_checkboxes: bool = False,
        dataset_callback: Optional[DatasetCallback] = None,
    ):
        """
        Initialize the dialog and set up the ui.

        Args:
            parent: Parent window.
            section: Section in the model to display and edit values for.
            get_curve: A callable that takes a curve ID and whether that curve uses dates, and returns the curve's
                X and Y values. The X values will be of type `np.datetime64` if using dates, or `float` otherwise.
                This can be `None` if there are no curve parameters.
            add_curve: A callable that takes a list of X values, a list of Y values, and whether to use dates, and
                returns a curve ID for a curve with the input values. X values will be of type `np.datetime64` if
                using dates, or `float` otherwise. This function should arrange so that calling `get_curve` later with
                the same curve ID and `use_dates` will return the same X and Y values passed to this function. This can
                be `None` if there are no curve parameters.
            is_interior: Whether the feature is on the interior of a coverage. If True, only groups that are legal on
                interior features will be shown.
            dlg_name: Unique name for this dialog. site-packages import path would make sense.
                Used as registry key for loading and saving the dialog geometry. Also used as a lookup key to find a
                help article when the Help button is clicked.
            window_title: Title of window.
            multi_select_message: A warning message to show the user. If nonempty, will be displayed prominently.
                Used by some models to display a warning when the user selects multiple features and tries to edit
                them all.
            banner: If nonempty, a tuple of [message, style sheet]. The message is displayed at the top of the dialog.
                The style sheet is how it's displayed, e.g. 'QLabel { background-color : rgb(200, 200, 150); }', or
                something that can be passed to QLabel.setStyleSheet().
            show_groups: Whether to show the group panel. If there is always exactly one group, this panel is just
                visual noise and can be disabled with this parameter.
            enable_unchecked_groups: Whether to allow setting parameters in groups that are unchecked. If `show_groups`
                is `False` or `hide_checkboxes` is `True`, this parameter is ignored and groups are always enabled.
            hide_checkboxes: Whether checkboxes are hidden from the left side groups, regardless of other settings.
            dataset_callback: A callback for picking a dataset or getting its label. See `DatasetCallback` for details.
                This can be `None` if there are no dataset parameters.
        """
        super().__init__(parent, dlg_name)
        self.setupUi(self)
        self.add_button.setIcon(QIcon(get_resource_path(':/resources/icons/add.svg')))
        self.remove_button.setIcon(QIcon(get_resource_path(':/resources/icons/delete.svg')))
        self.edit_button.setIcon(QIcon(get_resource_path(':/resources/icons/materials_edit.svg')))

        if hide_checkboxes or not show_groups:
            # Disabling unchecked groups makes the dialog somewhat useless when there's no way to check them.
            enable_unchecked_groups = True

        self._dataset_callback = dataset_callback
        self.section = section
        self._get_curve = get_curve
        self._add_curve = add_curve
        self.is_interior = is_interior
        self.hide_checkboxes = hide_checkboxes

        title = window_title if window_title else 'Map Feature Options'
        self.setWindowTitle(title)
        self._enable_unchecked_groups = enable_unchecked_groups

        # group_name (as in GroupSet.group(group_name)) -> widget containing parameter widgets
        self._group_pages: dict[str | int, QWidget] = {}
        # group_name (as in GroupSet.group(group_name)) -> list item widget
        self._list_widget_items: dict[str | int, QxListWidgetItem] = {}

        self._init_multi_select_message(multi_select_message)
        self._init_banner(banner)
        self._populate_tree()
        self.group_list.itemChanged.connect(self._item_changed)
        self.group_list.currentTextChanged.connect(self._group_changed)
        if not show_groups:
            self.group_list.hide()
            for i in range(self.splitter.count()):
                self.splitter.handle(i).setEnabled(False)

        self.group_buttons_container.hide()  # These are just here for a subclass to use.
        shortcut = QShortcut(QKeySequence.Find, self)
        shortcut.activated.connect(self.find_widget)

    def find_widget(self):
        """Ask the user for a parameter ID and then highlight its widget."""
        text, _ = QInputDialog.getText(self, 'Find a parameter widget', 'Name:')
        children = self.children()
        palette = self.palette()
        found_child = None

        while children:
            child = children.pop()
            children.extend(child.children())
            if not isinstance(child, ParameterWidget):
                continue
            if child.parameter_name == text:
                found_child = child
            else:
                child.setPalette(palette)

        selected_group_widget = self.group_list.currentItem()
        current_group = selected_group_widget.group_id

        if not found_child:
            QMessageBox.information(self, 'Widget not found', 'Widget not found')
        elif found_child.group_name != current_group:
            QMessageBox.information(self, 'Widget hidden', f'Widget is in another group: {found_child.group_name}')
        elif found_child.gmi_enabled:
            palette = QPalette()
            palette.setColor(QPalette.Window, Qt.cyan)
            found_child.setAutoFillBackground(True)
            found_child.setPalette(palette)
            self.scroll_area.ensureWidgetVisible(found_child)
        else:
            parent = found_child.gmi_parent.parameter_name
            QMessageBox.information(self, 'Widget hidden', f'Widget is hidden because of {parent}')

    def _init_multi_select_message(self, multi_select_message: str) -> None:
        """Initializes the multi-select warning message.

        Args:
            multi_select_message (str): If multiple feature objects selected, the message to show to warn the user.
        """
        if multi_select_message:
            self.txt_multiple_selections.setText(multi_select_message)
            self.txt_multiple_selections.setVisible(True)
        else:
            self.txt_multiple_selections.setVisible(False)

    def _init_banner(self, banner: Optional[tuple[str, str]]) -> None:
        """Initializes the banner, if there is one.

        Args:
            banner: Message displayed at the top of the dialog.
        """
        if banner and isinstance(banner, tuple) and len(banner) == 2:
            message, style_sheet = banner
            self.txt_banner.setText(message)
            self.txt_banner.setStyleSheet(style_sheet)
            self.txt_banner.setVisible(True)
        else:
            self.txt_banner.setVisible(False)

    def showEvent(self, event):  # noqa: N802 - function name should be lowercase
        """
        Restore last position and geometry when showing dialog.

        :meta private:  This only looks public because it overrides something in Qt.
        """
        super().showEvent(event)
        self._restore_splitter_geometry()

    def _package_name(self) -> str:
        """Uses self._dlg_name to return the package name (e.g. 'xms.hgs')."""
        parts = self._dlg_name.split('.')
        package_parts = parts[:2]
        if len(package_parts) == 2:
            return '.'.join(package_parts)
        else:
            return ''

    def _save_splitter_geometry(self) -> None:
        """Save the current position of the splitter."""
        settings = SettingsManager()
        settings.save_setting(self._package_name(), f'{self._dlg_name}.splitter', self.splitter.sizes())

    def _restore_splitter_geometry(self) -> None:
        """Restore the position of the splitter."""
        splitter = self._get_splitter_sizes()
        if not splitter:
            if self.group_list.isVisible():
                self.splitter.setSizes([100, 200])  # Default the right to be 2 times as wide as the left
            return

        splitter_sizes = [int(size) for size in splitter]
        self.splitter.setSizes(splitter_sizes)

    def _get_splitter_sizes(self) -> tuple[int, int]:
        """Returns a list of the splitter sizes that are saved in the registry."""
        settings = SettingsManager()
        splitter = settings.get_setting(self._package_name(), f'{self._dlg_name}.splitter')
        # get_setting decides its return type at runtime, so it's defined to return `object`.
        # We always store a tuple into this, so we always get a tuple back.
        return splitter  # type: ignore[return-value]

    def _populate_tree(self):
        """Populate the group tree."""
        added_group = False
        first_checked_item = None
        active_groups = set(self.section.active_group_names)
        for index, group_name in enumerate(self.section.group_names):
            group = self.section.group(group_name)
            if self.is_interior and not group.legal_on_interior:
                continue

            group_item = QxListWidgetItem(group_name, group.label, self.group_list)
            self._list_widget_items[group_name] = group_item
            checked = group_name in active_groups
            if not self.hide_checkboxes:
                group_item.setCheckState(Qt.Checked if checked else Qt.Unchecked)
            if checked and first_checked_item is None:
                first_checked_item = index
            if group.description:
                # Make it rich text so it will get word wrapped (see https://stackoverflow.com/questions/4795757)
                group_item.setToolTip(f'<span>{group.description}</span>')
            added_group = True

        if added_group:
            self.group_list.setCurrentRow(first_checked_item if first_checked_item is not None else 0)
            self._load_param_widgets()

    def _load_param_widgets(self):
        """Loads parameter widgets into the scrollable area."""
        selected_group_widget = self.group_list.currentItem()
        group_name = selected_group_widget.group_id
        if group_name not in self._group_pages:
            self._create_group_page(group_name)

        self.page_container.setCurrentWidget(self._group_pages[group_name])
        self._enable_page()

    def _create_group_page(self, group_name):
        """
        Create the group page and child widgets for a group.

        Args:
            group_name: Name of the group to create widgets for.
        """
        parent_widget = QWidget()

        parent_widget.setLayout(QVBoxLayout())
        self.page_container.addWidget(parent_widget)
        self._group_pages[group_name] = parent_widget

        make_widgets(self._dataset_callback, parent_widget, group_name, self.section.group(group_name))

    def _group_changed(self, _):
        """
        Tree selection changed slot.

        This is called when a new group in the group list is selected.
        It is *not* called when a group is checked/unchecked.
        """
        self._load_param_widgets()
        self._enable_page()

    def _item_changed(self, item: QxListWidgetItem):
        """
        Handle when a group list item's check state is changed.

        Args:
            item: The item that changed.
        """
        # Note that QListWidget.itemChanged is raised for basically anything anyone might conceivably care about,
        # so we have to be careful about behaving well on false alarms.
        if self.section.exclusive_groups and item.checkState() == Qt.Checked:
            for row in range(self.group_list.count()):
                checked_item = self.group_list.item(row)
                if checked_item != item and not self.hide_checkboxes:
                    checked_item.setCheckState(Qt.Unchecked)
        if item != self.group_list.currentItem():
            self.group_list.setCurrentItem(item)
        self._enable_page()

    def _enable_page(self):
        """Enables/disables the right side based on checked state of group and enable_uncheck_groups."""
        selected_group_widget = self.group_list.currentItem()
        if not self._enable_unchecked_groups:
            self.page_container.currentWidget().setEnabled(selected_group_widget.checkState() == Qt.Checked)

    def _get_checked_groups(self) -> set[str]:
        """Returns the set of groups that are checked."""
        checked_groups = set()  # Names of checked groups
        for i in range(self.group_list.count()):
            child = self.group_list.item(i)
            if child.checkState() == Qt.Checked:
                checked_groups.add(child.group_id)
        return checked_groups

    def _missing_required_inputs(self) -> bool:
        """Returns True if any required inputs are missing values."""
        missing_required = ''
        checked_groups = self._get_checked_groups()
        for group_name, page in self._group_pages.items():
            if group_name not in checked_groups:
                continue
            children = page.children()
            for child_widget in children:
                if isinstance(child_widget, ParameterWidget):
                    parameter = child_widget.parameter
                    # How to check if the value is "empty" may depend on the type of parameter so maybe should be a
                    # method on the parameter, but since we're only doing this with TEXT parameters so far, I'm going
                    # to just handle it here.
                    if parameter.required and (parameter.value is None or parameter.value == ''):
                        missing_required += f'\n{group_name}, {child_widget.parameter.label}'

        if missing_required:
            msg = f'Some required inputs are missing values:{missing_required}'
            ok_btn = self.button_box.button(QDialogButtonBox.Ok)
            pos = ok_btn.mapToGlobal(QPoint(ok_btn.width(), -ok_btn.height()))  # Offset tooltip to not be under mouse
            QToolTip.showText(pos, msg)
        return missing_required != ''

    def accept(self):
        """
        Handle the user clicking OK.

        :meta private:  This is only public because it's a slot. User code shouldn't use it.
        """
        if self._missing_required_inputs():
            return

        self._save_splitter_geometry()

        group_list = self.group_list
        for i in range(group_list.count()):
            child = group_list.item(i)
            group_name = child.group_id
            checked = child.checkState() == Qt.Checked
            self.section.group(group_name).is_active = checked
        super().accept()
