"""TimeSeriesFileReader class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import logging
from pathlib import Path
import re
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.xy.xy_series import XySeries

# 4. Local modules
from xms.gssha.file_io import io_util


def read(file_path: str | Path, start_date_time: datetime) -> dict[int, XySeries]:
    """Reads the TIME_SERIES_FILE (.xys) file and returns a dict of the xy series.

    See https://www.gsshawiki.com/File_Formats:Time_and_Elevation_Series_Files

    Args:
        file_path: Time series file path.
        start_date_time: Starting date/time from START_DATE, START_TIME in .gssha file.

    Returns:
        See description.
    """
    logger = logging.getLogger('xms.gssha')
    logger.info('Reading time series (.xys) file...')
    xy_series_dict: dict[int, XySeries] = {}
    next_id = 1  # Series IDs to use if we can't get them from the name (if from WMS, we should be able to)
    with open(file_path, 'r') as file:
        for line in file:
            if line.startswith('GSSHA_TS'):
                xy_series = read_time_series(file, start_date_time, next_id)
                xy_series_dict[xy_series.series_id] = xy_series
                if xy_series.series_id == next_id:
                    next_id += 1
    return xy_series_dict


def read_time_series(file: TextIO, start_date_time: datetime, series_id: int) -> XySeries:
    """Reads the time series from the file and returns it as an XySeries.

    Absolute dates/times are converted to relative using start_date_time.

    Args:
        file: The opened file.
        start_date_time: Starting date/time from START_DATE, START_TIME in .gssha file.
        series_id: Series ID to use if we can't get it from the name.

    Returns:
        See description.
    """
    line = next(file).rstrip('\n')
    name: str = line.strip('"')

    # If it's a time series file (.xys), and written by WMS, we should be able to get the series ID, which is
    # important because we'll need it when we read the WMS .map file and need to look up the xy series by ID.
    m = re.match(r'^ts_([0-9]+)$', name)  # e.g. "ts_123"
    if m:
        series_id = int(m.group(1))

    x = []
    y = []
    for line in file:
        if line.startswith('END_TS'):
            return XySeries(x, y, name, series_id, use_dates_times=False)
        elif line.startswith('ABSOLUTE'):
            pass
        else:
            line = line.rstrip('\n')
            pos_last_space = line.rfind(' ')
            date_time = io_util.datetime_from_string(line[:pos_last_space])

            # Convert from date/time to relative time in minutes using start_date_time
            minutes = io_util.relative_time_from_datetime(date_time, start_date_time)

            x.append(minutes)
            y_val = float(line[pos_last_space + 1:])
            y.append(y_val)
