"""Dialog for selecting dataset tree items."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QMessageBox

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util
from xms.api.tree import TreeNode

# 4. Local modules
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.treeitem_selector_datasets import TreeItemSelectorDatasetsDlg


class DatasetSelector:
    """A class for selecting a dataset and setting a label to the dataset name."""
    @staticmethod
    def select_dataset(
        parent,
        label_widget,
        caption,
        pe_tree,
        condition,
        storage_dict,
        key,
        icon_method,
        query=None,
        selected_time_step=0,
        allow_all_time_steps=False,
        allow_change_selection=True
    ):
        """Brings up the file selector dialog and sets the label_widget text if label_widget exists.

        Args:
            parent (QWidget): The parent widget for the dialog.
            label_widget (QLabel): The label widget that stores the dataset name.
            caption (str): Some sort of caption.
            pe_tree (TreeNode): Project explorer tree node.
            condition (function): A function to return true if the item is valid.
            storage_dict (dict): A dictionary that stores the result.
            key (str): The key in the dictionary for storing the result.
            icon_method: Method for TreeItemSelectorDlg to call to get the geom icon.
            query (xms.api.dmi.Query): Object for communicating with GMS. If not None, the
                dialog that appears will allow for selecting time steps (query is needed for that).
            selected_time_step (int): Index of the selected time step.
            allow_all_time_steps (bool): If true (and query is not None), an 'All time steps' toggle is
                displayed.
            allow_change_selection (bool): Whether to allow changing the selected dataset. If this is True,
                only the time step can be selected for the selected dataset.

        Returns:
            (bool): True if dialog accepted

        """
        tree_copy = TreeNode(other=pe_tree)  # Create a copy of the project explorer tree to filter.
        if tree_copy:
            tree_util.filter_project_explorer(tree_copy, condition)
        if not tree_copy.children:  # Have a geometry but no selectable datasets
            # This could happen if the filter removes all the datasets, e.g. because it filters to transient vector
            # datasets and the user selected a geometry that only has steady-state scalar dataset children.
            msg = QMessageBox(QMessageBox.Information, 'Error', 'No selectable datasets found.', QMessageBox.Ok, parent)
            msg.exec_()
            return False

        if not query:  # Show dialog without the time step list
            dlg = TreeItemSelectorDlg(
                caption,
                xmd.DatasetItem,
                tree_copy,
                storage_dict[key],
                override_icon=icon_method,
                show_root=True,
                parent=parent
            )
            if dlg.exec_():
                dset_uuid = dlg.get_selected_item_uuid()
                if label_widget:
                    if dset_uuid:
                        dset_path = tree_util.build_tree_path(tree_copy, dset_uuid)
                        label_widget.setText(dset_path)
                    else:
                        label_widget.setText('(none selected)')
                storage_dict[key] = dset_uuid
                return True
        else:  # Show dialog with the time step list
            dlg = TreeItemSelectorDatasetsDlg(
                title=caption,
                pe_tree=tree_copy,
                selected_dataset=storage_dict[key],
                selected_time_step=selected_time_step,
                query=query,
                allow_all_time_steps=allow_all_time_steps,
                override_icon=icon_method,
                show_root=True,
                parent=parent,
                allow_change_selection=allow_change_selection
            )
            if dlg.exec_():
                dset_uuid = dlg.get_selected_item_uuid()
                if dset_uuid:
                    dset_path = tree_util.build_tree_path(tree_copy, dset_uuid)
                    ts = dlg.get_selected_time_step_string()
                    if label_widget:
                        label_widget.setText(f'{dset_path}{ts}')
                    storage_dict[key] = f'{dset_uuid}{ts}'
                else:
                    storage_dict[key] = ''
                    if label_widget:
                        label_widget.setText('(none selected)')
                return True
        return False

    @staticmethod
    def is_scalar_if_dset(item):
        """Check if a tree item is scalar but only if it is a dataset.

        Args:
            item (TreeNode): The item to check

        Returns:
            (bool): True if the tree item is scalar or is not a dataset.
        """
        if type(item.data) is xmd.DatasetItem:
            if item.num_components == 1:
                return True
            return False
        return True

    @staticmethod
    def is_vector_if_dset(item):
        """Check if a tree item is a vector dataset but only if it is a dataset.

        Args:
            item (TreeNode): The item to check

        Returns:
            (bool): True if the tree item is a vector dataset or is not a dataset.
        """
        if type(item.data) is xmd.DatasetItem:
            if item.num_components == 2:
                return True
            return False
        return True

    @staticmethod
    def is_scalar_and_steady_state_if_dset(item):
        """Check if a tree item is steady state and scalar but only if it is a dataset.

        Args:
            item (TreeNode): The item to check

        Returns:
            (bool): True if the tree item is scalar and steady state or is not a dataset.
        """
        if type(item.data) is xmd.DatasetItem:
            if item.num_components == 1 and item.num_components == 1:
                return True
            return False
        return True
