"""Dialog for showing the map locator."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import contextlib
import os
import sys
import winreg

# 2. Third party modules
import folium
from folium.plugins import Geocoder, MousePosition
from PySide2.QtCore import QUrl
from PySide2.QtWidgets import QApplication

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.guipy.data.map_locator_web_parameters import ParametersObject, ParametersWebPage
from xms.guipy.dialogs.map_locator_dialog_ui import Ui_MapLocatorDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

VE_KEY_ROOT = 'Software\\ViaVirtualEarth\\WinformsEarth'


class MapLocatorDialog(XmsDlg):
    """Dialog for showing the map locator dialog."""
    def __init__(self, output_file: str, parent=None):
        """Initializes the dialog, sets up the ui.

        Args:
            output_file: Output filename for passing data back to XMS from this dialog
            parent (:obj:`QWidget`): Parent window
        """
        super().__init__(parent, 'xms.guipy.dialogs.map_locator_dialog')

        self.setWindowTitle('Map Locator')

        temp_dir = XmEnv.xms_environ_process_temp_directory()
        self._url_file = os.path.join(temp_dir, f'map_locator_{os.getpid()}.html')
        self._output_file = output_file
        self._viewer = None
        self._latitude = None
        self._longitude = None
        self._zoom = None
        self._x_min = None
        self._x_max = None
        self._y_min = None
        self._y_max = None
        self._bounds = None
        self.ui = Ui_MapLocatorDialog()
        self._setup_ui()

    def _get_starting_location_and_zoom(self):
        reg_key = None
        with contextlib.suppress(EnvironmentError):
            reg_key = winreg.OpenKey(winreg.HKEY_CURRENT_USER, VE_KEY_ROOT)
        if reg_key is None:
            reg_key = winreg.CreateKey(winreg.HKEY_CURRENT_USER, VE_KEY_ROOT)
        lat = 0.0
        lon = 0.0
        zoom = 2
        with contextlib.suppress(EnvironmentError, ValueError, IndexError):
            lat_str = winreg.QueryValueEx(reg_key, 'bm_last_lat')[0]
            reg_lat = float(lat_str)
            lon_str = winreg.QueryValueEx(reg_key, 'bm_last_lon')[0]
            reg_lon = float(lon_str)
            zoom_str = winreg.QueryValueEx(reg_key, 'bm_last_zoom')[0]
            reg_zoom = int(zoom_str)
            lat = reg_lat
            lon = reg_lon
            zoom = reg_zoom
        return [lat, lon], zoom

    def _save_location_to_registry(self):
        if self._latitude is not None:
            winreg.CreateKey(winreg.HKEY_CURRENT_USER, VE_KEY_ROOT)
            reg_key = winreg.OpenKey(winreg.HKEY_CURRENT_USER, VE_KEY_ROOT, access=winreg.KEY_WRITE)
            winreg.SetValueEx(reg_key, 'bm_last_lat', 0, winreg.REG_SZ, f"{self._latitude}")
            winreg.SetValueEx(reg_key, 'bm_last_lon', 0, winreg.REG_SZ, f"{self._longitude}")
            winreg.SetValueEx(reg_key, 'bm_last_zoom', 0, winreg.REG_SZ, f"{self._zoom}")

    def _save_location_to_file(self):
        if self._output_file is not None and self._y_max is not None:
            with open(self._output_file, 'w') as output:
                output.write('YMax:\n')
                output.write(f'{self._y_max}\n')
                output.write('YMin:\n')
                output.write(f'{self._y_min}\n')
                output.write('XMax:\n')
                output.write(f'{self._x_max}\n')
                output.write('XMin:\n')
                output.write(f'{self._x_min}\n')
                output.write('END\n')

    def _setup_ui(self):
        """Sets up the UI."""
        self.ui.setupUi(self)

        # set up the action preview window
        self._setup_viewer()

        # connect the slots
        self._connect_slots()
        self._update_map()

    def _setup_viewer(self):
        """Sets up the model."""
        self._viewer = ParametersWebPage(self)
        # self._viewer.setContextMenuPolicy(Qt.ContextMenuPolicy.NoContextMenu)
        self.ui.vlay_map.addWidget(self._viewer)

    def _connect_slots(self):
        """Connect Qt widget signal/slots."""
        self.ui.buttonBox.accepted.connect(self.accept)
        self.ui.buttonBox.rejected.connect(self.reject)

    def _update_map(self):
        """Update the map."""
        # add the map
        location, zoom = self._get_starting_location_and_zoom()
        folium_map = folium.Map(tiles='', location=location, zoom_start=zoom)
        folium.TileLayer('OpenStreetMap', name='Open Street Map', attr='OpenStreetMap', show=False).add_to(folium_map)
        folium.TileLayer('Esri.WorldStreetMap', name='ESRI World Street Map', attr='ESRI',
                         show=False).add_to(folium_map)
        folium.TileLayer('Esri.WorldImagery', name='ESRI World Imagery', attr='ESRI', show=False).add_to(folium_map)
        folium.TileLayer('Esri.WorldTopoMap', name='ESRI World Topo Map', attr='ESRI', show=False).add_to(folium_map)
        url = 'http://server.arcgisonline.com/ArcGIS/rest/services/USA_Topo_Maps/MapServer/tile/{z}/{y}/{x}'
        folium.TileLayer(tiles=url, name='ESRI USA Topo Map', attr='ESRI', show=False).add_to(folium_map)
        folium.TileLayer('Esri.OceanBasemap', name='ESRI Ocean Base Map', attr='ESRI', show=False).add_to(folium_map)
        folium.TileLayer('Esri.NatGeoWorldMap', name='ESRI National Geographic World Map', attr='ESRI',
                         show=False).add_to(folium_map)
        url = 'http://mt.google.com/vt/lyrs=y&x={x}&y={y}&z={z}'
        folium.TileLayer(tiles=url, name='Google Hybrid', attr='Google').add_to(folium_map)
        url = 'http://mt.google.com/vt/lyrs=m&hl=en&x={x}&y={y}&z={z}'
        folium.TileLayer(tiles=url, name='Google Map', attr='Google', show=False).add_to(folium_map)
        url = 'http://mt.google.com/vt/lyrs=s&hl=en&x={x}&y={y}&z={z}'
        folium.TileLayer(tiles=url, name='Google Satellite', attr='Google', show=False).add_to(folium_map)
        url = 'http://mt.google.com/vt/lyrs=p&hl=en&x={x}&y={y}&z={z}'
        folium.TileLayer(tiles=url, name='Google Terrain Streets Water', attr='Google', show=False).add_to(folium_map)
        Geocoder(provider='arcgis').add_to(folium_map)
        folium.LayerControl().add_to(folium_map)
        formatter = "function(num) {return L.Util.formatNum(num, 3) + ' &deg; ';};"
        MousePosition(
            position="bottomleft",
            separator=" | ",
            empty_string="NaN",
            lng_first=True,
            num_digits=20,
            prefix="Coordinates:",
            lat_formatter=formatter,
            lng_formatter=formatter,
        ).add_to(folium_map)
        folium_map.save(self._url_file)
        self._map_name = folium_map.get_name()
        self._viewer.map_name = self._map_name
        parameters = ParametersObject("parameters", self._map_name, self)
        self._viewer.page().add_object(parameters)
        self._viewer.setUrl(QUrl.fromLocalFile(self._url_file))
        self._viewer.show()

    def set_parameters(self, lat, long, zoom, x_min, x_max, y_min, y_max):
        """Set the parameters as a callback from the javascript web page code."""
        self._latitude = lat
        self._longitude = long
        self._zoom = zoom
        self._x_min = x_min
        self._x_max = x_max
        self._y_min = y_min
        self._y_max = y_max

    def accept(self):
        """Save data from dialog to Windows registry on OK."""
        self._save_location_to_registry()
        self._save_location_to_file()
        self._viewer.close()
        if os.path.exists(self._url_file):
            os.remove(self._url_file)
        super().accept()

    def reject(self):
        """Save data from dialog on OK."""
        self._save_location_to_registry()
        self._viewer.close()
        if os.path.exists(self._url_file):
            os.remove(self._url_file)
        super().reject()


if __name__ == '__main__':
    # Create the Qt Application
    app = QApplication(sys.argv)
    # Create and show the form
    form = MapLocatorDialog(sys.argv[1] if len(sys.argv) > 1 else None)
    # Run the main Qt loop
    form.exec()
