"""The SRH-2D plots dialog."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (QDialogButtonBox, QHBoxLayout, QVBoxLayout)

# 3. Aquaveo modules
from xms.guipy.data.plot_and_table_data_base import PlotsAndTableDataBase
from xms.guipy.dialogs.plot_and_table_widget import np_arrays_from_file as np2, PlotsAndTableWidget
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules


def np_arrays_from_file(file_name):
    """Read data in the files into numpy arrays.

    Args:
        file_name (str): file name

    Returns:
        (tuple(list[array_names], numpy 2d array)): output
    """
    return np2(file_name)


class PlotsAndTableDialog(XmsDlg):
    """A dialog for showing SRH-2D Output plots."""
    def __init__(self, win_cont, data: PlotsAndTableDataBase, help_url: str = None):
        """Initializes the dialog, sets up the ui.

        Args:
            win_cont (QWidget): Parent dialog
            data (PlotsAndTableDataBase): data class to populate plot
            help_url (str, optional): URL for help documentation
        """
        super().__init__(win_cont, 'xms.guipy.dialogs.PlotsAndTableDialog')
        self.widgets = {}
        self.plots_table_widget = PlotsAndTableWidget(self, data)

        self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Plots'
        if help_url is not None:
            self.help_url = help_url

        window_title = data.window_title
        self.setWindowTitle(window_title)
        self._setup_ui()

    def _setup_ui(self) -> None:
        """Sets up the dialog controls."""
        # Make the main layout a vertical layout
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])
        self.widgets['main_vert_layout'].addWidget(self.plots_table_widget)  # Pass in arguments

        self._setup_ui_bottom_button_box()

    def _setup_ui_bottom_button_box(self) -> None:
        """Add buttons to the bottom of the dialog."""
        # Add Import and Export buttons
        self.widgets['btn_horiz_layout'] = QHBoxLayout()
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(QDialogButtonBox.Close | QDialogButtonBox.Help)
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['btn_box'].helpRequested.connect(self._help_requested)
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_box'])
        self.widgets['main_vert_layout'].addLayout(self.widgets['btn_horiz_layout'])

    def _help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)
