"""MergedSvgIconEngine class."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QRect, QSize, Qt
from PySide2.QtGui import QIcon, QIconEngine, QPainter, QPixmap
from PySide2.QtSvg import QSvgRenderer

# 3. Aquaveo modules

# 4. Local modules


class MergedSvgIconEngine(QIconEngine):
    """A QIconEngine that merges two SVG icons together."""
    def __init__(self, base_icon: str, overlay_icon: str):
        """Initializes the class.

        Args:
            base_icon: First icon.
            overlay_icon: Second icon.
        """
        super().__init__()
        self.base_icon: str = base_icon
        self.overlay_icon: str = overlay_icon

    def clone(self) -> 'MergedSvgIconEngine':
        """Returns a clone of this object."""
        return MergedSvgIconEngine(self.base_icon, self.overlay_icon)

    def paint(self, painter: QPainter, rect: QRect, mode: QIcon.Mode, state: QIcon.State) -> None:
        """Paints the icon.

        Args:
            painter: The painter.
            rect: The rect.
            mode: The mode.
            state: The state.
        """
        # Draw base icon
        renderer = QSvgRenderer(self.base_icon)
        renderer.setAspectRatioMode(Qt.KeepAspectRatio)
        renderer.render(painter, rect)

        # Draw overlay icon
        renderer = QSvgRenderer(self.overlay_icon)
        renderer.setAspectRatioMode(Qt.KeepAspectRatio)
        renderer.render(painter, rect)

    def pixmap(self, size: QSize, mode: QIcon.Mode, state: QIcon.State) -> QPixmap:
        """Returns the pixmap.

        Args:
            size: Size of the pixmap.
            mode: Mode of the icon.
            state: State of the icon.

        Returns:
            The pixmap.
        """
        pixmap = QIcon(self.base_icon).pixmap(size, mode, state)
        painter = QPainter(pixmap)
        rect = QRect(0, 0, size.width(), size.height())
        self.paint(painter, rect, mode, state)
        return pixmap
