"""Base class for Qt table widgets used by XMS modules."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel
from PySide2.QtWidgets import QHBoxLayout, QHeaderView, QSizePolicy, QSpacerItem, QToolBar, QVBoxLayout, QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_table_view import QxTableView


class BasicTableWidget(QWidget):
    """Table widget base class."""
    def __init__(self, parent):
        """Construct the widget.

        Args:
            parent (Something derived from QObject): The parent object.
        """
        super().__init__(parent)
        self.table_view = None
        self.toolbar = None
        self.btn_actions = None
        self.layout = QVBoxLayout()
        self.model = None
        self.toolbar_layout = None
        self.movable_rows = False
        self.add_icon = ':resources/icons/row-add.svg'
        self.delete_icon = ':resources/icons/row-delete.svg'
        self.row_up_icon = ':resources/icons/row-up.svg'
        self.row_down_icon = ':resources/icons/row-down.svg'
        self.use_header_checkboxes: bool = False
        self.allow_drag_fill = False  # Set this to True to allow for drag-fill of cells like Excel (kind of).

        # Set this to False if you don't want to set columns with combobox delegates as combobox columns in the model.
        self.set_cbx_columns_in_model = True

    def _setup_ui(self, column_delegates, stretch_last_section, fixed_size, filter_model=None, movable_rows=False):
        """Add the table widget and initialize the model.

        Args:
            column_delegates (dict): Key is column index, value is delegate
            stretch_last_section (bool): If True, last section will stretch to fill table
            fixed_size (bool): If False, toolbar with '+' and 'x' buttons will be added above the table
            filter_model (QSortFilterProxyModel): Filter proxy model to install
            movable_rows (bool): True if rows can be reordered
        """
        self.table_view = QxTableView(self)
        self.table_view.set_cbx_columns_in_model = self.set_cbx_columns_in_model
        self.table_view.allow_drag_fill = self.allow_drag_fill
        self.table_view.use_header_checkboxes = self.use_header_checkboxes
        self.table_view.setModel(self.model)
        self.movable_rows = movable_rows
        # Add proxy model if there is one
        if filter_model is not None:
            self.table_view.filter_model = filter_model
            self.table_view.filter_model.setSourceModel(self.model)
            self.table_view.setModel(self.table_view.filter_model)
        # Add column delegates
        for col, delegate in column_delegates.items():
            self.table_view.setItemDelegateForColumn(col, delegate)
        self.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)
        self.table_view.horizontalHeader().setStretchLastSection(stretch_last_section)
        widget_builder.resize_columns_to_contents(self.table_view)
        # Add the "+", "-" buttons if table is not fixed size.
        if not fixed_size:
            self.table_view.selectionModel().selectionChanged.connect(self.on_selection_changed)
            self.toolbar = QToolBar()
            button_list = [
                [self.add_icon, 'Add row', self.on_btn_add_row],
                [self.delete_icon, 'Delete row', self.on_btn_delete_row],
            ]
            # Add the up and down buttons if rows can be reordered.
            if self.movable_rows:
                button_list.extend(
                    [
                        [self.row_up_icon, 'Move up', self.on_btn_up],
                        [self.row_down_icon, 'Move down', self.on_btn_down],
                    ]
                )
            self.btn_actions = widget_builder.setup_toolbar(self.toolbar, button_list)
            # enable_delete = self.model.rowCount() > 0
            # self.toolbar.widgetForAction(self.btn_actions[self.delete_icon]).setEnabled(enable_delete)
            # Set the layout
            self.toolbar_layout = QHBoxLayout()
            spacer = QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum)
            self.toolbar_layout.addWidget(self.toolbar)
            self.toolbar_layout.addSpacerItem(spacer)
            self.layout.addLayout(self.toolbar_layout)
            self.on_selection_changed()  # Initialize state of the toolbar buttons
        elif fixed_size:
            self.table_view.resize_height_to_contents()
        self.layout.addWidget(self.table_view)
        self.setLayout(self.layout)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        pass

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        pass

    def on_btn_up(self):
        """Called when the Up button is clicked. Moves rows up in the table."""
        selected_list = self.table_view.selectionModel().selectedIndexes()
        if not selected_list:
            return

        self._move_row(True, selected_list)

    def on_btn_down(self):
        """Called when the Down button is clicked. Moves rows down in the table."""
        selected_list = self.table_view.selectionModel().selectedIndexes()
        if not selected_list:
            return

        self._move_row(False, selected_list)

    def on_selection_changed(self):
        """Called when the user clicks in the table and changes what cells are selected."""
        selection_list = self.table_view.selectedIndexes()
        selections_exist = len(selection_list) > 0
        empty_table = (self.model.rowCount() == 0)
        enable_actions = selections_exist and not empty_table

        self.toolbar.widgetForAction(self.btn_actions[self.delete_icon]).setEnabled(enable_actions)
        if self.movable_rows:
            self.toolbar.widgetForAction(self.btn_actions[self.row_up_icon]).setEnabled(enable_actions)
            self.toolbar.widgetForAction(self.btn_actions[self.row_down_icon]).setEnabled(enable_actions)

    def _move_row(self, up, selected_list):
        """Moves a row up or down.

        Note: This implementation only works with DataFrames that have a 1-based sequential integer Index. If your
              DataFrame is not structured this way, you will need to override this method.

        Args:
            up (bool): True if moving up, else False
            selected_list (list[PySide2.QtCore.QModelIndex]): The list of selected cells
        """
        source_row = selected_list[0].row()
        source_idx = source_row + 1  # Assuming a 1-based sequential integer pandas.Index
        dest_row = source_row - 1 if up else source_row + 1
        dest_idx = dest_row + 1  # Assuming a 1-based sequential integer pandas.Index

        self.model.swap_rows(source_idx, dest_idx, source_row, dest_row)

        # Update the selection
        new_index = self.model.index(dest_row, selected_list[0].column())
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )
