"""CheckBoxHeader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QPoint, QRect, Qt, Signal
from PySide2.QtGui import QMouseEvent, QPainter
from PySide2.QtWidgets import QHeaderView, QStyle, QStyleOptionButton, QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel


class CheckBoxHeader(QHeaderView):
    """A QHeaderView supporting checkboxes to the left of the text of a subset of columns.

    Adapted from: https://stackoverflow.com/a/30938728/5666265
    """

    # Signals
    clicked = Signal(int, Qt.CheckState)

    # Constants
    X_OFFSET = 3  # margin in x
    WIDTH = 20  # width of checkbox
    HEIGHT = 20  # height of checkbox

    def __init__(
        self, column_idxs: list[int], orientation: Qt.Orientation = Qt.Horizontal, parent: QWidget | None = None
    ):
        """Initializer.

        Args:
            column_idxs: List of column indices (0-based) that will have checkboxes.
            orientation: The orientation.
            parent: The parent.
        """
        super().__init__(orientation, parent)

        if not isinstance(column_idxs, list):
            raise ValueError('column_idxs must be a list')

        self._column_idxs = column_idxs
        self._is_checked: dict[int, Qt.CheckState] = {column: Qt.Unchecked for column in self._column_idxs}
        self._y_offset = 0  # This will be calculated based on the height of the paint rect
        self._x_offset = self.X_OFFSET  # This will be calculated if we are centering the checkboxes
        self._read_only_columns = set()
        self.center_checkboxes = False

        self.setSectionsClickable(True)
        self.setSectionResizeMode(QHeaderView.Stretch)

    def paintSection(self, painter: QPainter, rect: QRect, logical_index: int) -> None:  # noqa: N802 - lowercase
        """Paints the section specified by the given logical_index, using the given painter and rect.

        Args:
            painter: The painter.
            rect: The rect.
            logical_index: The logical index.
        """
        painter.save()
        super().paintSection(painter, rect, logical_index)
        painter.restore()

        # Compute y offset
        self._y_offset = int((rect.height() - self.HEIGHT) / 2.)
        # Compute x offset if centering checkboxes, or use the constant margin value otherwise.
        if self.center_checkboxes:
            self._x_offset = int((rect.width() - self.WIDTH) / 2.)
        else:
            self._x_offset = self.X_OFFSET

        if logical_index in self._column_idxs:
            # Draw the checkbox
            option = QStyleOptionButton()
            option.rect = self._checkbox_rect(rect.x(), rect.y())
            if logical_index not in self._read_only_columns:
                option.state = QStyle.State_Enabled | QStyle.State_Active
            else:
                option.state = QStyle.State_ReadOnly

            if self._is_checked[logical_index] == Qt.PartiallyChecked:
                option.state |= QStyle.State_NoChange
            elif self._is_checked[logical_index]:
                option.state |= QStyle.State_On
            else:
                option.state |= QStyle.State_Off
            self.style().drawControl(QStyle.CE_CheckBox, option, painter)

    def mousePressEvent(self, event: QMouseEvent) -> None:  # noqa: N802 - should be lowercase
        """Handle mouse press event.

        Args:
            event: The event
        """
        index = self.logicalIndexAt(event.pos())
        if 0 <= index < self.count():
            if index in self._column_idxs and self._in_checkbox(event.pos()) and index not in self._read_only_columns:
                if self._is_checked[index] == Qt.Checked:
                    self._is_checked[index] = Qt.Unchecked
                else:
                    self._is_checked[index] = Qt.Checked

                self.clicked.emit(index, self._is_checked[index])
                self.viewport().update()
            else:
                super(CheckBoxHeader, self).mousePressEvent(event)
        else:
            super(CheckBoxHeader, self).mousePressEvent(event)

    def sectionSize(self, logical_index: int) -> int:  # noqa: N802 - should be lowercase
        """Return the section size.

        Args:
            logical_index: The logical index.

        Returns:
            See description.
        """
        width = super().sectionSize(logical_index)
        return width + self._checkbox_width_with_margin()

    def _checkbox_width_with_margin(self) -> int:
        """Return the width of the checkbox, including the margin on the left and right."""
        return self.WIDTH + (2 * self.X_OFFSET)

    def update_check_state(self, logical_index: int, state: Qt.CheckState) -> None:
        """Updates the state of the checkbox for the section specified by logical_index.

        Args:
            logical_index: The logical index of the section.
            state: The state.
        """
        self._is_checked[logical_index] = state
        self.viewport().update()

    def check_state(self, logical_index: int) -> Qt.CheckState:
        """Return the state of the checkbox at the logical_index.

        Args:
            logical_index:

        Returns:
            See description.
        """
        return self._is_checked[logical_index]

    def set_read_only_columns(self, read_only_columns: set[int]) -> None:
        """Set which columns are read only."""
        self._read_only_columns = read_only_columns

    def _in_checkbox(self, point: QPoint) -> bool:
        """Return true if point is inside the checkbox.

        Args:
            point: The point.

        Returns:
            See description.
        """
        index = self.logicalIndexAt(point)
        rect = self._section_checkbox_rect(index)
        return rect.contains(point)

    def _section_checkbox_rect(self, logical_index: int) -> QRect:
        """Return the QRect of the checkbox.

        Assumes self._y_offset has been computed in paintSection().

        Args:
            logical_index: The logical index of the section.

        Returns:
            See description.
        """
        x = self.sectionPosition(logical_index)
        return self._checkbox_rect(x, 0)

    def _checkbox_rect(self, x: int, y: int) -> QRect:
        """Return the QRect of the checkbox.

        Assumes self._y_offset has been computed in paintSection().

        Args:
            x: x at top left of section.
            y: y at top left of section.

        Returns:
            See description.
        """
        return QRect(x + self._x_offset, y + self._y_offset, self.WIDTH, self.HEIGHT)


def update_header_from_data(model: QxPandasTableModel, header: CheckBoxHeader) -> None:
    """Update the state of the checkboxes in the header based on the model data.

    Args:
        model: The model.
        header: The CheckBoxHeader.
    """
    for col_idx in range(model.columnCount()):
        checked = False
        unchecked = False
        for row_idx in range(model.rowCount()):
            state = model.data(model.index(row_idx, col_idx), role=Qt.CheckStateRole)
            if state == Qt.Checked:
                checked = True
            elif state == Qt.Unchecked:
                unchecked = True

        if checked and unchecked:
            header.update_check_state(col_idx, Qt.PartiallyChecked)
        elif checked:
            header.update_check_state(col_idx, Qt.Checked)
        else:
            header.update_check_state(col_idx, Qt.Unchecked)


def update_data_from_header(logical_index: int, state: Qt.CheckState, model: QxPandasTableModel) -> None:
    """Update model data (checkbox column) from header checkbox state.

    Typically called by slot connected to 'clicked' signal emitted by CheckBoxHeader.

    Args:
        logical_index: The logical index of the section clicked.
        state: The state.
        model: The model.
    """
    for row_idx in range(model.rowCount()):
        model.setData(model.index(row_idx, logical_index), state, Qt.CheckStateRole)
