"""Widget for the point symbol selector."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSize, Qt, Signal
from PySide2.QtGui import QColor, QIcon, QImage, QPainter, QPixmap
from PySide2.QtSvg import QGraphicsSvgItem
from PySide2.QtWidgets import QGridLayout, QPushButton, QSizePolicy

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.point_symbol import PointSymbol
import xms.guipy.resources.guipy  # noqa: F401, F403
from xms.guipy.widgets.button_enum import ButtonEnumWidget


class PointSymbolWidget(ButtonEnumWidget):
    """A widget showing point symbols on buttons in a grid in a given color and size."""
    symbol_order = [
        PointSymbol.PT_SQUARE, PointSymbol.PT_CIRCLE, PointSymbol.PT_TRI, PointSymbol.PT_DIAMOND,
        PointSymbol.PT_SQUARE_H, PointSymbol.PT_CIRCLE_H, PointSymbol.PT_TRI_H, PointSymbol.PT_DIAMOND_H,
        PointSymbol.PT_CROSS, PointSymbol.PT_X, PointSymbol.PT_SURVEY
    ]
    symbol_svg = {
        PointSymbol.PT_CIRCLE: ":resources/symbols/circle.svg",
        PointSymbol.PT_CROSS: ":resources/symbols/cross.svg",
        PointSymbol.PT_DIAMOND: ":resources/symbols/diamond.svg",
        PointSymbol.PT_CIRCLE_H: ":resources/symbols/hollow_circle.svg",
        PointSymbol.PT_DIAMOND_H: ":resources/symbols/hollow_diamond.svg",
        PointSymbol.PT_SQUARE_H: ":resources/symbols/hollow_square.svg",
        PointSymbol.PT_TRI_H: ":resources/symbols/hollow_triangle.svg",
        PointSymbol.PT_SQUARE: ":resources/symbols/square.svg",
        PointSymbol.PT_SURVEY: ":resources/symbols/survey.svg",
        PointSymbol.PT_TRI: ":resources/symbols/triangle.svg",
        PointSymbol.PT_X: ":resources/symbols/x.svg",
    }
    symbol_changed = Signal()

    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from QObject): The parent object.
        """
        super().__init__(parent)
        self.layout = QGridLayout(self)
        self.buttons = {}
        row = 0
        col = 0
        for symbol in self.symbol_order:
            btn = QPushButton(self)
            btn.setMinimumSize(QSize(50, 50))
            btn.setMaximumSize(QSize(150, 150))
            btn.setSizeIncrement(1, 1)
            btn.setCheckable(True)
            btn.setAutoExclusive(True)
            btn.clicked.connect(self.symbol_changed)
            self.layout.addWidget(btn, row, col)
            self.buttons[symbol] = btn
            col += 1
            if col > 3:
                col = 0
                row += 1

        self.setSizePolicy(QSizePolicy.Policy.Preferred, QSizePolicy.Policy.Preferred)

    def set_symbol(self, symbol):
        """Checks the button showing the symbol.

        Args:
            symbol (PointSymbol): The symbol that is to be checked.
        """
        self.buttons[symbol].setChecked(True)

    def get_symbol(self):
        """Gets the symbol of the currently checked button.

        Returns:
            symbol (PointSymbol): The symbol.
        """
        for symbol in self.symbol_order:
            if self.buttons[symbol].isChecked():
                return symbol

    def get_current_icon(self, color, size, btn_size):
        """Gets the icon drawn from the current symbol, in the specified color and size.

        Args:
            color (QColor): The current color.
            size (int): The current pixel size.
            btn_size (int): The size of the button to draw on. Assumes square.

        Returns:
            icon (QIcon): The icon.
        """
        return self.get_icon(self.symbol_svg[self.get_symbol()], color, size, btn_size)

    @staticmethod
    def get_icon(path, color, size, btn_size):
        """Gets the icon drawn from the svg file, in the specified color and size.

        Args:
            path (str): Relative path to the *.svg resource file.
            color (QColor): The current color.
            size (int): The current pixel size.
            btn_size (int): The size of the button to draw on. Assumes square.

        Returns:
            icon (QIcon): The icon.
        """
        if size <= 4:
            # same as self.symbol_svg[PointSymbol.PT_SQUARE]
            svg = QGraphicsSvgItem(":resources/square.svg")
        else:
            svg = QGraphicsSvgItem(path)
        svg_img = QImage(btn_size, btn_size, QImage.Format_ARGB32_Premultiplied)
        svg_img.fill(0x00000000)
        svg_painter = QPainter(svg_img)
        svg.renderer().render(svg_painter)
        svg_painter.end()
        svg_pix = QPixmap.fromImage(svg_img.scaled(size, size))
        mask = svg_pix.createMaskFromColor(QColor('black'), Qt.MaskOutColor)
        svg_pix.fill(color, btn_size)
        svg_pix.setMask(mask)
        svg_icon = QIcon(svg_pix)
        return svg_icon

    def draw_all_symbols(self):
        """Draws all of the different symbols on buttons, in the black and size 20."""
        color = QColor('black')
        size = 20
        for symbol in self.symbol_order:
            self.buttons[symbol].setIcon(
                self.get_icon(self.symbol_svg[symbol], color, size, self.buttons[PointSymbol.PT_CIRCLE].size().width())
            )
            self.buttons[symbol].setIconSize(QSize(size, size))
