"""FileImporter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import shutil

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs
from xms.guipy.dialogs import dialog_util, process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.package_reader_base import PackageReaderBase
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.misc import log_util


def import_file(filepath: Pathlike, data: BaseFileData, win_cont: QWidget) -> BaseFileData:
    """Import a file.

    Args:
        filepath: The file being imported
        data: The data being overwritten.
        win_cont: The window container.

    Returns:
        The new data.
    """
    thread = FileImportFeedbackThread(filepath, data)
    process_feedback_dlg.run_feedback_dialog(thread, win_cont)
    return thread.new_data


class FileImportFeedbackThread(FeedbackThread):
    """Thread for importing a file."""
    def __init__(self, filepath: Pathlike, data: BaseFileData):
        """Initializes the class.

        Args:
            filepath: The file being imported
            data: The data being overwritten.
        """
        super().__init__(is_export=False, create_query=False)

        self._filepath = filepath
        self._data = data
        self.new_data = None
        self.display_text |= {
            'title': 'Import MODFLOW 6 file',
            'working_prompt': f'Importing MODFLOW 6 file \"{self._filepath}\".',
            'error_prompt': 'Error(s) encountered while importing.',
            'warning_prompt': 'Warning(s) encountered while importing.',
            'success_prompt': f'Successfully imported \"{self._filepath}\".',
        }

    def _run(self) -> None:
        """Does the work."""
        orig_dir = Path(self._data.filename).parent
        try:
            with dialog_util.wait_cursor_context():
                log_util.clear_log_capture()

                # Read
                reader = io_factory.reader_from_ftype(self._data.ftype)
                new_data = reader.read(self._filepath, mfsim=self._data.mfsim, model=self._data.model)
                if not new_data:
                    raise RuntimeError()

                # Change new data filename to original location
                new_data.filename = self._data.filename

                # Read the settings from the original location
                reader.read_settings()

                # Write
                _clear_dir(orig_dir, ['settings.json'])
                self._write(new_data)
                if log_util.errors_logged():
                    raise RuntimeError()
                self.new_data = new_data
        except Exception:
            # Restore from the backup dir, if we can
            backup_dir = f'{orig_dir}_backup'
            if Path(backup_dir).is_dir():
                fs.clear_folder(orig_dir)
                shutil.copytree(backup_dir, orig_dir, dirs_exist_ok=True)

    def _write(self, new_data: BaseFileData) -> None:
        """Write new data to the new directory.

        Args:
            new_data: The imported data.
        """
        mfsim_dir = Path(self._data.mfsim.filename).parent
        components_dir = mfsim_dir.parent
        writer_options = WriterOptions(mfsim_dir=str(mfsim_dir), use_open_close=True, dmi_sim_dir=str(components_dir))
        PackageReaderBase.importing = True
        new_data.write(writer_options)
        PackageReaderBase.importing = False


def _clear_dir(dir_path: Pathlike, keep: list | None = None) -> None:
    """Recursively delete everything in dir_path except files specified by keep_suffixes or keep_names.

    Args:
        dir_path: Path of directory to clear.
        keep: Optional list of names or suffixes of files or folders to keep, e.g. ['.db', 'settings.json', 'subdir'].
    """
    for item in dir_path.iterdir():
        if item.is_file():
            if not keep or (item.suffix not in keep and item.name not in keep):
                try:
                    item.unlink()
                except OSError as e:
                    print(e)
        elif item.is_dir():
            if not keep or (item.suffix not in keep and item.name not in keep):
                _clear_dir(item, keep)  # Recurse
                if not os.listdir(item):  # Remove the directory if it is now empty
                    shutil.rmtree(item)


def _copy_new_to_orig(new_dir: Pathlike, orig_dir: Pathlike) -> None:
    fs.clear_folder(orig_dir)
    shutil.copytree(new_dir, orig_dir, dirs_exist_ok=True)
