"""run_new_sim_dialog function."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread

# 4. Local modules
from xms.mf6.components.component_creator import ComponentCreator
from xms.mf6.components.default_package_creator import DefaultPackageCreator
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.gui.new_sim_dialog import NewSimDialog
from xms.mf6.misc import log_util


def run_dialog(main_file: str, query, params, win_cont: QWidget, model_str, help_id, dlg_data=None):
    """Opens the New Simulation dialog and creates the model files indicated.

    Args:
        main_file (str): The main file associated with this component.
        query (xmsapi.dmi.Query): Object for communicating with GMS
        params: Generic map of parameters. Unused in this case.
        win_cont: The window container.
        model_str (str): If coming from Add Package > GWF (or GWT), is 'GWF' or 'GWT' respectively
        help_id (str): Identifier string to lookup the URL for the help button in the dialog
        dlg_data (dict): the data from the dialog

    Returns:
        (tuple): tuple containing:
            - messages (list of tuple of str): List of tuples with the first element of the
              tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
              text.
            - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
    """
    del params  # Unused parameter
    messages = []
    actions = []

    new_sim_or_model_uuid = query.current_item_uuid() if model_str else query.parent_item_uuid()
    dialog = NewSimDialog(query.project_tree, new_sim_or_model_uuid, win_cont, model_str, help_id)
    dialog.setModal(True)
    dialog_result = dialog.exec()
    if dialog_result != QDialog.Accepted:
        query.delete_item(new_sim_or_model_uuid)
        return messages, actions

    data = dlg_data if dlg_data is not None else dialog.data

    # Make sure we have a UGrid
    if not data['ugrid_uuid']:
        query.delete_item(new_sim_or_model_uuid)
        messages.append(('ERROR', 'No UGrid was selected. Action aborted.'))
        return messages, actions

    worker = CreationWorker(main_file, query, model_str, data, new_sim_or_model_uuid)
    thread = ProcessFeedbackThread(do_work=worker.do_work, parent=win_cont)
    model_or_sim_str = 'model' if model_str else 'simulation'
    display_text = {
        'title': f'{model_or_sim_str.title()} Creation',
        'working_prompt': f'Creating the {model_or_sim_str}...',
        'error_prompt': f'Error(s) encountered while creating the {model_or_sim_str}.',
        'warning_prompt': f'Warning(s) encountered while creating the {model_or_sim_str}.',
        'success_prompt': f'Successfully created the {model_or_sim_str}.',
        'auto_load': 'Close this dialog automatically when finished.',
        'log_format': '%(asctime)s - %(message)s',
        'use_colors': True
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, worker=thread, parent=win_cont)
    feedback_dlg.testing = True
    feedback_dlg.exec()
    if not worker.completed:
        query.delete_item(new_sim_or_model_uuid)
    return messages, actions


class CreationWorker:
    """Creates the new sim or model, for use in ProcessFeedbackThread."""
    def __init__(self, main_file: str, query: Query, model_str: str, data: dict, new_sim_or_model_uuid: str) -> None:
        """Initializer.

        Args:
            main_file: The main file associated with this component.
            query: Object for communicating with GMS
            model_str: If coming from Add Package > GWF (or GWT), is 'GWF' or 'GWT' respectively
            data: the data from the dialog
            new_sim_or_model_uuid: Uuid of the new simulation or the new model
        """
        self._main_file = main_file
        self._query = query
        self._model_str = model_str
        self._data = data
        self._new_sim_or_model_uuid = new_sim_or_model_uuid
        self.completed = False

    def do_work(self) -> None:
        """Creates the sim or model."""
        dogrid = self._query.item_with_uuid(self._data['ugrid_uuid'])

        # Create all packages
        creator = DefaultPackageCreator()
        dir_ = os.path.dirname(self._main_file)
        mfsim, messages = creator.create_simulation(dir_, self._data, dogrid, self._model_str)
        if messages:
            self._query.delete_item(self._new_sim_or_model_uuid)
            log = log_util.get_logger()
            log.error('Error creating simulation.')
            return

        creator = ComponentCreator()
        sim_uuid = creator.create_components_from_simulation(
            mfsim, self._query, ugrid_uuid=self._data['ugrid_uuid'], model_str=self._model_str
        )

        if self._model_str:  # Adding a model ('GWF6', 'GWT6' etc)
            _rewrite_mfsim_nam(sim_uuid, self._query)

        self.completed = True


def _rewrite_mfsim_nam(sim_uuid: str, query: Query) -> None:
    """Read and write the mfsim.nam file.

    Args:
        sim_uuid: Tree node uuid of simulation.
        query: Object for communicating with GMS.
    """
    # Read
    sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
    mfsim_nam = query.item_with_uuid(
        item_uuid=sim_node.uuid, model_name="MODFLOW 6", unique_name="Sim_Manager"
    ).main_file
    reader = io_factory.reader_from_ftype('MFSIM6')
    mfsim = reader.read(mfsim_nam, sim_node, query)

    # Write
    dmi_sim_dir = str(Path(mfsim_nam).parent.parent)
    mfsim_dir = str(Path(mfsim_nam).parent)
    writer_options = WriterOptions(mfsim_dir=mfsim_dir, dmi_sim_dir=dmi_sim_dir, just_name_file=True)
    mfsim.write(writer_options)
