"""CsubData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util, units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class CsubData(GriddataBase):
    """Data class to hold the info from the CSUB package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'CSUB6'
        self.add_block('GRIDDATA', ['CG_SKE_CR', 'CG_THETA', 'SGM', 'SGS'])
        self.block_with_cellids = 'PACKAGEDATA'
        self.period_files = {}  # Dict of 1-based stress periods and the paths to their external files
        self.list_blocks = {'PACKAGEDATA': ''}  # (list blocks -> filename)
        self.period_files = {}  # Dict of 1-based stress periods and the paths to their external files

    def get_units(self, array_name: str) -> str:
        """Returns a units string.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        units_str = ''
        match array_name:
            case 'CG_SKE_CR':  # specific storage or recompression index if COMPRESSION_INDICES option is used
                if not self.options_block.get('COMPRESSION_INDICES'):
                    units_str = '[L^-1]'
            case 'CG_THETA':
                units_str = units_util.UNITS_UNITLESS  # Porosity
            case 'SGM':
                units_str = units_util.UNITS_UNITLESS  # specific gravity
            case 'SGS':
                units_str = units_util.UNITS_UNITLESS  # specific gravity
            case _:
                return ''  # This is an error
        return units_str

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return False

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() in {'CG_SKE_CR', 'CG_THETA'}

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        if block.upper() == 'PACKAGEDATA':
            id_columns = data_util.get_id_column_dict(self.grid_info())
            columns = {
                **{
                    'ICSUBNO': (np.int32, 1),
                },
                **id_columns,
                **{
                    'CDELAY': (object, ''),
                    'PCS0': (np.float64, 0.0),
                    'THICK_FRAC': (np.float64, 0.0),
                    'RNB': (np.float64, 0.0),
                    'SSV_CC': (np.float64, 0.0),
                    'SSE_CR': (np.float64, 0.0),
                    'THETA': (np.float64, 0.0),
                    'KV': (np.float64, 0.0),
                    'H0': (np.float64, 0.0),
                }
            }
            data_util.add_boundname_columns_to_dict(self.options_block, columns)

        else:  # This would be the stress periods
            return self.package_column_info()

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
        k_units = units_util.string_from_units(self, units_util.UNITS_K)
        if block.upper() == 'PACKAGEDATA':
            return {
                names.index('ICSUBNO'): 'Interbed number',
                names.index('CDELAY'): 'Subsidence delay type',
                names.index('PCS0'):
                    f'Initial offset from the calculated initial effective stress or initial'
                    f' preconsolidation stress in the interbed {length_units}',
                names.index('THICK_FRAC'): 'Interbed thickness or cell fraction of the interbed.',
                names.index('RNB'):
                    'Interbed material factor equivalent number of interbeds in the interbed'
                    ' system represented by the interbed.',
                names.index('SSV_CC'): 'Initial inelastic specific storage or compression index.',
                names.index('SSE_CR'):
                    'Initial elastic coarse-grained material specific storage or recompression'
                    ' index.',
                names.index('THETA'): 'Initial porosity of the interbed.',
                names.index('KV'): f'Vertical hydraulic conductivity of the delay interbed {k_units}',
                names.index('H0'):
                    f'Initial offset from the head in cell cellid or the initial head in the delay'
                    f' interbed {length_units}',
            }
        else:
            return {names.index('SIG0'): 'Stress offset for the cell.'}

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        id_columns = data_util.get_id_column_dict(self.grid_info())
        columns = {
            **id_columns,
            **{
                'SIG0': (object, ''),
            }
        }

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Skeletal Storage, Compaction, and Subsidence (CSUB) Package'

    def get_time_series_columns(self) -> list[int]:
        """Returns a list of the column indices that can contain time series.

        Returns:
            List of indices of columns that can contain time series.
        """
        names, _, _ = self.get_column_info('')
        start = names.index('SIG0')
        return [start]  # This is the sig0 column

    # def read_file_into_dataframe(self, filename, block):
    #     """Reads the file into a dataframe and returns the dataframe.
    #
    #     Returns:
    #         (DataFrame): The dataframe.
    #     """
    #     column_names, column_types, _ = self.get_column_info(block)
    #     return gui_util.read_csv_file_into_dataframe(filename, column_names, column_types)

    def block_with_boundnames(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    def stress_id_columns(self):
        """Returns the column name where the id exists that can be used to help identify this stress across periods.

        Typically is 'CELLIDX' which is added by GMS but is 'RNO' for SFR.

        Returns:
            See description.
        """
        return ['CELLIDX']

    # def map_info(self, feature_type):
    #     """Returns info needed for Map from Coverage.
    #
    #     Args:
    #         feature_type (str): 'points', 'arcs', or 'polygons'
    #
    #     Returns:
    #         (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
    #     """
    #     if feature_type != 'polygons':
    #         return {}
    #     return {'Name': None, 'SS': None, 'SY': None}

    # def append_period_data(self, period, rows):
    #     """Appends the data to the exiting period info.
    #
    #     Args:
    #         period (int): Stress period.
    #         rows (list): List of lists consisting of rows of data.
    #     """
    #     database = Database(self)
    #     database.append_period_data(period, rows)

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxField('GAMMAW', brief='Unit weight of water', type_='float', value=9806.5),
                CheckboxField('BETA', brief='Compressibility of water', type_='float', value=4.6512e-10),
                Checkbox('HEAD_BASED', brief='Head-based formulation will be used'),
                Checkbox('INITIAL_PRECONSOLIDATION_HEAD', brief='Preconsolidation heads will be specified'),
                CheckboxField(
                    'NDELAYCELLS', brief='Number of nodes used to discretize delay interbeds', type_='int', value=0
                ),
                Checkbox('COMPRESSION_INDICES', brief='Recompression (CR) and compression (CC) indices are specified'),
                Checkbox(
                    'UPDATE_MATERIAL_PROPERTIES', brief='Thickness and void ratio will vary during the simulation'
                ),
                Checkbox('CELL_FRACTION', brief='Interbed thickness specified as fraction of cell thickness'),
                Checkbox(
                    'SPECIFIED_INITIAL_INTERBED_STATE',
                    brief='Absolute preconsolidation stresses (heads) and delay bed heads will be specified'
                ),
                Checkbox(
                    'SPECIFIED_INITIAL_PRECONSOLIDATION_STRESS',
                    brief='Absolute preconsolidation stresses (heads) will be specified for interbeds'
                ),
                Checkbox(
                    'SPECIFIED_INITIAL_DELAY_HEAD',
                    brief='Absolute initial delay bed head will be specified for interbeds'
                ),
                Checkbox(
                    'EFFECTIVE_STRESS_LAG',
                    brief='Effective stress from previous time step used to calculate specific storage'
                ),
                CheckboxField('STRAIN_CSV_INTERBED FILEOUT', brief='Save final interbed strain', type_='str'),
                CheckboxField(
                    'STRAIN_CSV_COARSE FILEOUT', brief='Save final coarse-grained material strain', type_='str'
                ),
                CheckboxField('COMPACTION FILEOUT', brief='Save compaction', type_='str'),
                CheckboxField('COMPACTION_ELASTIC FILEOUT', brief='Save elastic interbed compaction', type_='str'),
                CheckboxField('COMPACTION_INELASTIC FILEOUT', brief='Save inelastic interbed compaction', type_='str'),
                CheckboxField('COMPACTION_INTERBED FILEOUT', brief='Save interbed compaction', type_='str'),
                CheckboxField(
                    'COMPACTION_COARSE FILEOUT', brief='Save elastic coarse-grained material compaction', type_='str'
                ),
                CheckboxField('ZDISPLACEMENT FILEOUT', brief='Save z-displacement', type_='str'),
                CheckboxField('PACKAGE_CONVERGENCE FILEOUT', brief='Save convergence info to csv file', type_='str'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'CG_SKE_CR':
            txt = (
                'Initial elastic coarse-grained material specific storage or recompression index. The recompression'
                ' index is specified if COMPRESSION_INDICES is specified in the OPTIONS block. Specified or calculated'
                ' elastic coarse-grained material specific storage values are not adjusted from initial values if'
                ' HEAD_BASED is specified in the OPTIONS block.'
            )
        elif tab_text == 'CG_THETA':
            txt = 'Initial porosity of coarse-grained materials.'
        elif tab_text == 'SGM':
            txt = (
                'Specific gravity of moist or unsaturated sediments. If not specified, then a default value of 1.7 is'
                ' assigned.'
            )
        elif tab_text == 'SGS':
            txt = (
                'Specific gravity of saturated sediments. If not specified, then a default value of 2.0 is assigned.'
            )
        return txt
