"""EvtListData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class EvtListData(ListPackageData):
    """Data class to hold the info from an EVT list-based package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'EVT6'
        self.nseg = 1  # Number of ET segments

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
        et_rate_units = units_util.string_from_units(self, units_util.UNITS_INFILTRATION)
        tool_tips = {
            names.index('SURFACE'): f'Elevation of the ET surface {length_units}',
            names.index('RATE'): f'Maximum ET flux rate {et_rate_units}',
            names.index('DEPTH'): f'ET extinction depth {length_units}',
        }
        # yapf: disable
        if 'PXDP' in names:
            tool_tips[
                names.index('PXDP')
            ] = 'Proportion of the ET extinction depth at the bottom of a segment (dimensionless).'
        if 'PETM' in names:
            tool_tips[
                names.index('PETM')
            ] = 'Proportion of the maximum ET flux rate at the bottom of a segment (dimensionless).'
        if 'PETM0' in names:
            tool_tips[names.index('PETM0')] = (
                'Proportion of the maximum ET flux rate that will apply when head is at'
                ' or above the ET surface (dimensionless).'
            )
        # yapf: enable
        return tool_tips

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        column_names = ['SURFACE', 'RATE', 'DEPTH']
        types = {'SURFACE': object, 'RATE': object, 'DEPTH': object}
        default = {'SURFACE': 0.0, 'RATE': 0.0, 'DEPTH': 0.0}

        # Add the pxdp and petm columns if nseg > 1
        if self.nseg > 1:
            px_list = ['PXDP', 'PETM']
            for j in range(2):
                px = px_list[j]
                for i in range(self.nseg - 1):
                    name = f'{px}{i+1}'
                    column_names.append(name)
                    types[name] = object
                    default[name] = 0.0
        if self.options_block.has('SURF_RATE_SPECIFIED'):
            column_names.append('PETM0')
            types['PETM0'] = object
            default['PETM0'] = 0.0

        return column_names, types, default

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Evapotranspiration (EVT) Package'

    def use_surf_rate_specified(self):
        """Returns True if using the SURF_RATE_SPECIFIED option (EVT package only).

        The EVT package overrides this method.

        Returns:
            (bool): True if the EVT package, else False.
        """
        return True

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'polygons':
            return {}
        return {'Name': None, 'SURFACE': None, 'RATE': None, 'DEPTH': None}

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        if feature_type != 'polygons':
            return {}
        column_names, _, _ = self.package_column_info()
        info = {column_name: None for column_name in column_names}
        self._add_aux_and_boundname_info(info)
        return info

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                Checkbox('FIXED_CELL', brief='Do not reassign to underlying cell if cell is inactive'),
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                CheckboxField(
                    'AUXMULTNAME',
                    brief='Auxiliary variable used as multiplier of evapotranspiration rate',
                    type_='str'
                ),
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
                Checkbox('SURF_RATE_SPECIFIED', brief='Specify petm0', check_box_method='on_chk_surf_rate_specified'),
            ]
        )
