"""VscData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_defs import CheckboxComboBox, CheckboxField


class VscData(ListPackageData):
    """Data class to hold the info from a VSC package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'VSC6'
        # self.nviscspecies = 0  # number of species used in the viscosity equation of state
        self.list_blocks['PACKAGEDATA'] = ''

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        return self.package_column_info()

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        return {
            names.index('IVISCSPEC'): 'Species number',
            names.index('DVISCDC'):
                'Slope of the line defining the linear relationship between viscosity and'
                ' temperature or between viscosity and concentration, depending on the type'
                ' of species entered on each line. If the value of AUXSPECIESNAME entered'
                ' on a line corresponds to TEMPERATURE_SPECIES_NAME (in the OPTIONS block),'
                ' this value will be used when VISCOSITY_FUNC is equal to LINEAR (the'
                ' default) in the OPTIONS block. When VISCOSITY_FUNC is set to NONLINEAR,'
                ' a value for DVISCDC must be specified though it is not used.',
            names.index('CVISCREF'):
                'Reference temperature or reference concentration value used for this'
                ' species in the viscosity equation of state. If AUXSPECIESNAME entered on'
                ' a line corresponds to TEMPERATURE_SPECIES_NAME (in the OPTIONS block),'
                ' then CVISCREF refers to a reference temperature, otherwise it refers to'
                ' a reference concentration.',
            names.index('MODELNAME'):
                'Name of a GWT model used to simulate a species that will be used in the'
                ' viscosity equation of state. This name will have no effect if the'
                ' simulation does not include a GWT model that corresponds to this GWF'
                'model.',
            names.index('AUXSPECIESNAME'):
                'Name of an auxiliary variable in a GWF stress package that will be'
                ' used for this species to calculate the viscosity values. If a'
                ' viscosity value is needed by the Viscosity Package then it will'
                ' use the temperature or concentration values associated with this'
                ' AUXSPECIESNAME in the viscosity equation of state. For advanced'
                ' stress packages (LAK, SFR, MAW, and UZF) that have an associated'
                ' advanced transport package (LKT, SFT, MWT, and UZT), the'
                ' FLOW_PACKAGE_AUXILIARY_NAME option in the advanced transport'
                ' package can be used to transfer simulated temperature or'
                ' concentration(s) into the flow package auxiliary variable. In this'
                ' manner, the Viscosity Package can calculate viscosity values for'
                ' lakes, streams, multi-aquifer wells, and unsaturated zone flow'
                ' cells using simulated concentrations.',
        }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        columns = {
            'IVISCSPEC': (np.int32, 1),
            'DVISCDC': (np.float64, 0.0),
            'CVISCREF': (np.float64, 0.0),
            'MODELNAME': (object, ''),
            'AUXSPECIESNAME': (object, ''),
        }
        return gui_util.column_info_tuple_from_dict(columns)

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Viscosity (VSC) Package'

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        a234_brief = 'Empirical parameter for calculating viscosity using a nonlinear formulation'
        return OptionsBlock(
            [
                CheckboxField(
                    'VISCREF',
                    brief='Fluid reference viscosity used in the equation of state',
                    type_='float',
                    value=1.0
                ),
                CheckboxField(
                    'TEMPERATURE_SPECIES_NAME',
                    brief='Auxspeciesname in PACKAGEDATA that corresponds to the temperature species',
                    type_='str'
                ),
                CheckboxComboBox(
                    'THERMAL_FORMULATION',
                    brief='Which viscosity formulation to use for the temperature species',
                    items=['LINEAR', 'NONLINEAR'],
                    value='LINEAR'
                ),
                CheckboxField('THERMAL_A2', brief=a234_brief, type_='float', value=10.0),
                CheckboxField('THERMAL_A3', brief=a234_brief, type_='float', value=248.37),
                CheckboxField('THERMAL_A4', brief=a234_brief, type_='float', value=133.15),
                CheckboxField('VISCOSITY FILEOUT', brief='Save viscosity to a binary file', type_='str'),
            ]
        )
