"""PackageReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import database
from xms.mf6.file_io.package_reader_base import PackageReaderBase


class PackageReader(PackageReaderBase):
    """Reads a list package file."""
    def __init__(self, ftype):
        """Initializes the class.

        Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
        """
        super().__init__()
        self.ftype = ftype

    def _read_dimensions(self, line):
        """Reads the dimensions block.

        Args:
            line (str): line from file
        """
        words = line.split()
        if not words:
            self.invalid_line_in_file(line)
            return

        if words[0].upper() == 'MAXBOUND' and len(words) > 1:
            self._data.maxbound = int(words[1])
        elif words[0].upper() == 'NSEG' and len(words) > 1:
            self._data.nseg = int(words[1])

    def _on_end_block(self, block_name):
        """Save lists to external files when we finish reading a stress period.

        Args:
            block_name (str): block name from file
        """
        if block_name == 'PERIOD' and hasattr(self._data, 'period_files'):
            sp = self._curr_block_number
            if sp not in self._data.period_files:
                self._data.period_files[sp] = ''

            if sp == self._data.nper():
                self._stop_after_block = 'PERIOD'

            # Set the periods.db filename if one exists
            database.set_periods_db_filename(self._data)

        elif block_name == 'GRIDDATA' and PackageReaderBase.importing:
            self._data.block('GRIDDATA').dump_to_temp_files(PackageReaderBase.temp_file_list)

    def _read_period(self, line):
        """Read stress period data.

        Args:
            line (str): line from file
        """
        sp = self._curr_block_number
        self._data.period_files[sp] = self._read_list_block(line, 'PERIOD')
        return True

    def _read_connectiondata(self, line):
        """Read DISU connection data.

        Args:
            line (str): line from file

        Returns:
            bool : False see comment below
        """
        self._handle_array_block(line)
        return False  # There may be more properties to follow in the griddata block

    def _read_griddata(self, line):
        """Reads the griddata block.

        Args:
            line (str): A line from the file.

        Returns:
            bool : False see comment below
        """
        self._handle_array_block(line)
        return False  # There may be more properties to follow in the griddata block
