"""TextFileContextManager class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import distutils.text_file
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class TextFileContextManager:
    """Context manager for use with distutils.text_file.TextFile() which is used to skip comments and blank lines."""
    def __init__(self, filepath: str | Path):
        """Initializer.

        Args:
            filepath: File and path.
        """
        self._filepath = filepath
        self._text_file = None

    def __enter__(self):
        """Context manager enter.

        Returns:
            (distutils.text_file.TextFile): The TextFile object.
        """
        self._text_file = distutils.text_file.TextFile(filename=self._filepath, strip_comments=True, skip_blanks=True)
        return self._text_file

    def __exit__(self, exc_type, exc_value, exc_tb):
        """Context manager exit.

        Args:
            exc_type: Exception type.
            exc_value: Exception value.
            exc_tb: Exception traceback.
        """
        if self._text_file:
            self._text_file.close()
