"""AtsDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.widgets.list_block_table_widget import ListBlockTableWidget


class AtsDialog(PackageDialogBase):
    """A dialog used for the ATS package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'PERIODDATA']
        self.default_sections = ['PERIODDATA']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'PERIODDATA':
            self.setup_perioddata_section()
        else:
            super().setup_section(section_name)

    def setup_perioddata_section(self):
        """Sets up the PERIODDATA section."""
        section = 'PERIODDATA'
        self.add_group_box_to_scroll_area(section)

        # Table
        w = self.uix[section]['tbl_wgt'] = ListBlockTableWidget(self, self.dlg_input, '')
        self.uix[section]['layout'].addWidget(w)

        self.get_table_widget(section).load_list_block(section)

    # @overrides
    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.
        """
        return self.uix['PERIODDATA']['tbl_wgt']

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.save_list_blocks_to_temp()

    @classmethod
    def run_dialog_on_file(cls, filepath, base_file_data, locked, parent):
        """Function called when Edit button in List dialog is clicked.

        See ListDialog.on_btn_edit

        Args:
            filepath (str): Filepath of obs file.
            base_file_data (BaseFileData): BaseFileData object.
            locked (bool): True if locked.
            parent: Parent widget
        """
        from xms.mf6.components import dialog_runner
        dialog_runner.run_dialog_with_file(filepath, base_file_data, locked, parent, 'ATS6', AtsDialog)
