"""Module for reading and writing hydrodynamics datasets."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['write_datasets']

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter

# 4. Local modules


def write_datasets(datasets: list[tuple[DatasetReader, str]], where: str):
    """
    Write the hydrodynamic datasets to disk.

    Args:
        datasets: List of tuples of (dataset, path). The path is where in the file to write the dataset to.
        where: Path to the destination file.
    """
    append = False
    for dataset, path in datasets:
        write_dataset(dataset, path, where, append)
        append = True


def write_dataset(dataset: DatasetReader, dataset_path: str, where: str, append: bool):
    """
    Write a dataset to disk.

    Args:
        dataset: The dataset to write.
        dataset_path: Path to store the dataset in the destination file.
        where: Path to the destination file.
        append: Whether to append the dataset to an existing file, as opposed to deleting the file and overwriting it.
    """
    writer = DatasetWriter(
        h5_filename=where,
        name=dataset_path,
        dset_uuid=dataset.uuid,
        geom_uuid=dataset.geom_uuid,
        num_components=dataset.num_components,
        ref_time=dataset.ref_time,
        null_value=dataset.null_value,
        time_units=dataset.time_units,
        location=dataset.location,
        overwrite=not append,
    )
    writer.write_xmdf_dataset(dataset.times, dataset.values, dataset.activity)
