"""Cell monitor data."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.tool_core.table_definition import FloatColumnType, TableDefinition

# 4. Local modules
from xms.rsm.data import bc_val_data_def as bvdd
from xms.rsm.data import monitor_data_def as mdd

CANAL_BC_UNIFORM_FLOW = 'Uniform flow (<uniformflow>)'
CANAL_BC_JUNCTION_BLOCK = 'Structure at junction (<junctionblock>)'
CANAL_BC_TYPES = [CANAL_BC_UNIFORM_FLOW, CANAL_BC_JUNCTION_BLOCK]

WB_BASIN = 'Basin'
WB_CANAL = 'Canal'
WB_IMPOUNDMENT = 'Impoundment'
WB_LAKE = 'Lake'
WB_MESH_CELL = 'Mesh Cell'
WB_WCD_WATERBODY = 'WCD Waterbody'

WB_TYPES = [WB_BASIN, WB_CANAL, WB_IMPOUNDMENT, WB_LAKE, WB_MESH_CELL, WB_WCD_WATERBODY]


class _WaterMoverDataDef:
    """A class to define the water mover data for the generic model."""
    def __init__(self):
        self.generic_model = GenericModel()
        self._cur_feature = ''  # POINTS or ARCS
        self._pp = self.generic_model.point_parameters
        self._point_data_def()
        self._arc_data_def()

    def _point_data_def(self):
        """Defines the point data for the generic model."""
        self._pp = self.generic_model.point_parameters
        self._cur_feature = 'POINTS'
        self._feature_data_def()

    def _arc_data_def(self):
        """Defines the arc data for the generic model."""
        self._pp = self.generic_model.arc_parameters
        self._cur_feature = 'ARCS'
        self._feature_data_def()

    def _feature_data_def(self):
        self._canal_bc()
        self._single_control()
        self._genxweir()
        self._genxweirsub()
        self._vnotchbleeder()
        self._circularbleeder()
        self._rectbleeder()
        self._setflow()
        self._shunt()
        self._monitor()

    def _add_label_waterbodies(self, grp):
        """Adds label and waterbody IDs to the group."""
        grp.add_text(name='label', label='Label')
        if self._cur_feature == 'POINTS':
            grp.add_text(name='id1', label='Label of waterbody 1 <id1>')
            grp.add_text(name='id2', label='Label of waterbody 2 <id2>')
        else:
            grp.add_option(name='id1', label='Type of waterbody 1 <id1>', options=WB_TYPES, default=WB_MESH_CELL)
            grp.add_option(name='id2', label='Type of waterbody 2 <id2>', options=WB_TYPES, default=WB_MESH_CELL)

    def _canal_bc(self):
        grp = self._pp.add_group(group_name='canal_bc', label='Canal boundary condition')
        grp.add_text(name='label', label='Boundary condition label (<label>)', default='')
        grp.add_integer(name='bc_id', label='Boundary condition id (<bcID>)', default=-1)
        label = 'Boundary condition type (<networkbc>)'
        opt_par = grp.add_option(name='bc_type', label=label, default=CANAL_BC_TYPES[0], options=CANAL_BC_TYPES)

        flags = {o: False for o in CANAL_BC_TYPES}
        flags[CANAL_BC_UNIFORM_FLOW] = True
        p = grp.add_float(name='slope', label='Slope (<slope>)', default=0.0)
        p.add_dependency(opt_par, flags)

    def _single_control(self):
        grp = self._pp.add_group(group_name='single_control', label='Single control')
        self._add_label_waterbodies(grp)
        opts = ['waterbody 1', 'waterbody 2']
        grp.add_option(name='controlled_by', label='Controlled by', default=opts[0], options=opts)
        grp.add_float(name='cutoff', label='Cutoff value (<cutoff>)', default=0.0)
        grp.add_boolean(name='gravflow', label='Gravity flow', default=False)
        grp.add_boolean(name='revflow', label='Reverse flow', default=False)
        cols = [FloatColumnType(header='Head', default=0.0), FloatColumnType(header='Discharge', default=0.0)]
        td = TableDefinition(cols)
        default_vals = []
        grp.add_table(name='table', label='Head-discharge table', table_definition=td, default=default_vals)

    def _genxweir(self):
        grp = self._pp.add_group(group_name='genxweir', label='General weir')
        self._weir_properties(grp)

    def _genxweirsub(self):
        grp = self._pp.add_group(group_name='genxweirsub', label='General weir submerged')
        self._weir_properties(grp)

    def _weir_properties(self, grp):
        self._add_label_waterbodies(grp)
        label = 'Weir coefficient for forward flow (1->2) (<fcoeff>)'
        grp.add_float(name='fcoeff', label=label, default=0.0)
        label = 'Weir coefficient for reverse flow (2->1) (<bcoeff>)'
        grp.add_float(name='bcoeff', label=label, default=0.0)
        grp.add_float(name='crestelev', label='Crest elevation', default=0.0)
        grp.add_float(name='crestlen', label='Crest length', default=0.0)
        label = 'Exponent "a" for upstream depth over weir (<dpower>)'
        grp.add_float(name='dpower', label=label, default=1.0, low=0.9, high=1.9)
        label = 'Exponent "b" for water level difference (<spower>)'
        grp.add_float(name='spower', label=label, default=0.5, low=0.2, high=0.9)

    def _vnotchbleeder(self):
        grp = self._pp.add_group(group_name='vnotchbleeder', label='V-Notch bleeder')
        self._add_label_waterbodies(grp)
        label = 'Elevation of lowest point of bleeder opening'
        grp.add_float(name='invert', label=label, default=0.0, low=0.0)
        label = 'Elevaton of highest point of bleeder opening'
        grp.add_float(name='top', label=label, default=0.0, low=0.0)
        grp.add_float(name='angle', label='Angle of the "V" (degrees)', default=0.0, low=0.0, high=180.0)

    def _circularbleeder(self):
        grp = self._pp.add_group(group_name='circularbleeder', label='Circular bleeder')
        self._add_label_waterbodies(grp)
        grp.add_float(name='centroid', label='Elevation of bleeder centroid', default=0.0, low=0.0)
        grp.add_float(name='diameter', label='Diameter', default=0.0, low=0.0)

    def _rectbleeder(self):
        grp = self._pp.add_group(group_name='rectbleeder', label='Rectangular bleeder')
        self._add_label_waterbodies(grp)
        grp.add_float(name='centroid', label='Elevation of bleeder centroid', default=0.0, low=0.0)
        grp.add_float(name='height', label='Height', default=0.0, low=0.0)
        grp.add_float(name='width', label='Width', default=0.0, low=0.0)

    def _setflow(self):
        grp = self._pp.add_group(group_name='setflow', label='Set flow')
        self._add_label_waterbodies(grp)
        lab = 'Water mover flow'
        bvdd.add_bc_val_to_group(group=grp, label_replace=lab, skip_csv=True, skip_rc=True)

    def _shunt(self):
        grp = self._pp.add_group(group_name='shunt', label='Shunt')
        self._add_label_waterbodies(grp)
        grp.add_float(name='sconst', label='Conveyance (<sconst>)', default=0.0)
        lab = 'Head in waterbody 1 below which no flow occurs (<bottom>)'
        grp.add_float(name='bottom', label=lab, default=0.0)

    def _monitor(self):
        pp = self._pp
        outputs = [
            ('flow', 'Monitor flow (<flow>)'),
            ('prevflow', 'Monitor previous flow (<prevflow>)'),
            ('volume', 'Monitor volume (<volume>)'),
            ('control', 'Monitor control (<control>)'),
            ('maxflow', 'Monitor maximum flow (<maxflow>)'),
            ('wmflow', 'Monitor nonlinear flow (<wmflow>)'),
            ('flowiter', 'Monitor linear flow (<flowiter>)'),
            ('capacity', 'Monitor capacity (<capacity>)'),
        ]
        for name, label in outputs:
            gp = pp.add_group(group_name=name, label=label)
            mdd.add_output_to_group(gp)


def generic_model():
    """Gets a generic model for the boundary conditions coverage.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    wmdd = _WaterMoverDataDef()
    return wmdd.generic_model
