"""Class for writing canal coverage data to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.rsm.data import canal_data_def as cdd
from xms.rsm.file_io.bc_val_writer import BcValWriter


class CanalBcsWriter:
    """Writer class for the RSM control file."""
    def __init__(self, coverage_name, canal_data, arc_start_id, writer_data):
        """Constructor.

        Args:
            coverage_name (str): The name of the coverage
            canal_data (CanalData): The canal data
            arc_start_id (int): The arc start id
            writer_data (WriterData): Class with information needed to writer model input files.
        """
        self._wd = writer_data
        self._coverage_name = coverage_name
        self._canal_data = canal_data
        self._arc_start_id = arc_start_id
        self._all_arc_data = {}
        self._bc_method = {
            cdd.BC_INFLOW: self._segment_source,
            cdd.BC_SPEC_HEAD: self._segment_head,
            cdd.BC_GEN_HEAD: self._segment_ghb,
            cdd.BC_SPEC_HEAD_JUNC: self._junction_head,
        }
        self._arc_id = -1
        self._arc_data = None
        self._bc_type = None
        self._bc_label = ''
        self._bc_kcoeff = None
        self._bc_val_segment = None
        self._bc_val_slope = None
        self._bc_csv_filename = ''
        self._network_xml = None
        self._xml_atts = {}
        self._label_to_id = {}
        self._feature_id = ''
        self.bcs = []

    def add_arc_data(self, arc_id, arc_data):
        """Adds arc data to the writer.

        Args:
            arc_id (int): arc id
            arc_data (xms.gmi.generic_model.Group): arc data
        """
        if arc_data.group_name == 'default_canal':
            return
        label = arc_data.parameter('label').value
        self._label_to_id[label] = arc_id
        # if 'network_bc' not in arc_data.parameter_names:
        #     return
        bc_type = arc_data.parameter('network_bc').value
        if cdd.BC_NONE == bc_type:
            return
        self._all_arc_data[arc_id] = arc_data

    def write(self):
        """Write the canal boundary conditions."""
        keys = list(self._all_arc_data.keys())
        keys.sort()
        for arc_id in keys:
            cov_arc_id = arc_id - self._arc_start_id
            self._feature_id = f'arc id: "{cov_arc_id}" in coverage: "{self._coverage_name}"'
            self._arc_id = arc_id
            self._arc_data = self._all_arc_data[arc_id]
            self._load_bc_data()
            self._bc_method[self._bc_type]()

    def _load_bc_data(self):
        """Load the bc data."""
        self._bc_type = self._arc_data.parameter('network_bc').value
        self._bc_label = self._arc_data.parameter('network_bc_label').value
        self._bc_id = self._arc_data.parameter('network_bc_id').value
        self._bc_kcoeff = self._arc_data.parameter('network_bc_ghb_coeff').value
        self._xml_atts = {'id': f'{self._arc_id}'}
        if self._bc_label:
            self._xml_atts['label'] = self._bc_label
        if self._bc_id > 0:
            self._xml_atts['bcID'] = str(self._bc_id)
        self._bc_csv_filename = f'{self._coverage_name}_{self._arc_id}.csv'

    def _do_bc(self, bc_xml_name):
        """Write the bc xml element."""
        elem = Et.Element(bc_xml_name, self._xml_atts)
        self._wd.csv_writer.set_desired_filename(self._bc_csv_filename)
        bc_val_writer = BcValWriter(
            elem, self._arc_data, self._wd.csv_writer, self._wd.rule_curve_label_id, self._feature_id
        )
        bc_val_writer.write()
        self.bcs.append(elem)

    def _segment_source(self):
        """Write the segment source bc."""
        self._do_bc('segmentsource')

    def _segment_head(self):
        """Write the segment head bc."""
        self._do_bc('segmenthead')

    def _segment_ghb(self):
        """Write the segment ghb bc."""
        self._xml_atts['kcoeff'] = f'{self._bc_kcoeff}'
        self._do_bc('segmentghb')

    def _junction_head(self):
        """Write the junction head bc."""
        self._do_bc('junctionhead')
