"""Data class for a mapped tidal component."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules
import xarray as xr

# 3. Aquaveo modules

# 4. Local modules
from xms.schism.data.base_data import BaseData


class MappedTidalData(BaseData):
    """Manages a mapped BC coverage's data."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """
        Initialize the object.

        Args:
            main_file: Path to the component's main file.
        """
        super().__init__(main_file)

        self.info.attrs['SCHISM_FILE_TYPE'] = 'MAPPED_TIDES'
        self.commit()

    @property
    def domain_hash(self) -> str:
        """The hash of the domain the mapped coverage was made for."""
        return self.info.attrs.get('domain_hash', '')

    @domain_hash.setter
    def domain_hash(self, value: str):
        """The hash of the domain the mapped coverage was made for."""
        self.info.attrs['domain_hash'] = value

    @property
    def coverage_uuid(self) -> str:
        """The UUID of the coverage the tides were mapped against."""
        return self.info.attrs.get('coverage_uuid', '')

    @coverage_uuid.setter
    def coverage_uuid(self, value: str):
        """The UUID of the coverage the tides were mapped against."""
        self.info.attrs['coverage_uuid'] = value

    @property
    def properties(self) -> xr.Dataset:
        """The tidal constituent properties dataset."""
        return self._get_dataset('properties')

    @properties.setter
    def properties(self, value: xr.Dataset):
        """The tidal constituent properties dataset."""
        self._set_dataset(value, 'properties')

    @property
    def elevation(self) -> xr.Dataset:
        """The elevation forcing dataset."""
        return self._get_dataset('elevation')

    @elevation.setter
    def elevation(self, value: xr.Dataset):
        """The elevation forcing dataset."""
        self._set_dataset(value, 'elevation')

    @property
    def velocity(self) -> xr.Dataset:
        """The velocity forcing dataset."""
        return self._get_dataset('velocity')

    @velocity.setter
    def velocity(self, value: xr.Dataset):
        """The velocity forcing dataset."""
        self._set_dataset(value, 'velocity')

    @property
    def source(self) -> int:
        """
        The source of the data.

        One of the constants defined in xms.tides.data.tidal_data, and an index into that module's TDB_SOURCES list.
        """
        value = self.info.attrs.get('source', -1)
        return value

    @source.setter
    def source(self, value: int):
        """
        The source of the data.

        One of the constants defined in xms.tides.data.tidal_data, and an index into that module's TDB_SOURCES list.
        """
        self.info.attrs['source'] = value

    @property
    def _dataset_names(self) -> set[str]:
        """Names of datasets used by this data manager."""
        return {'properties', 'velocity', 'elevation'}

    @property
    def _main_file_name(self) -> str:
        """What to name the component's main file."""
        return 'schism_mapped_tides.nc'

    def _create_dataset(self, name: str):
        """Create a dataset."""
        if name in ['properties', 'velocity', 'elevation']:
            return xr.Dataset()

        raise AssertionError('Unknown dataset')  # pragma: nocover
