"""MaterialComponent class. Data for material Coverage."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import uuid

# 2. Third party modules
import pandas as pd
from PySide2.QtGui import QColor

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest  # noqa: I202
from xms.components.bases.coverage_component_base import ColAttType
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.polygon_texture import PolygonOptions, PolygonTexture
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.srh.components.srh_component import duplicate_display_opts
from xms.srh.components.srh_cov_component import SrhCoverageComponent
from xms.srh.data.material_data import DEFAULT_MATERIAL_COLORS, MaterialData
from xms.srh.gui.material_dialog import MaterialDialog

MAT_INITIAL_ATT_ID_FILE = 'initial_mat_polys.ids'
MAT_INITIAL_COMP_ID_FILE = 'initial_mat_comp.ids'


class MaterialComponent(SrhCoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.is_sediment = False
        self.data = MaterialData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Material List and Properties', 'open_material_properties'),
        ]
        self.polygon_commands = [('Assign Material', 'open_assign_poly_materials')]
        # Copy default display options if needed
        comp_dir = os.path.dirname(self.main_file)
        self.disp_opts_file = os.path.join(comp_dir, 'material_display_options.json')
        if not os.path.exists(self.disp_opts_file):
            # Copy over the unassigned material category id file.
            default_id_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                f'mat_{MaterialData.UNASSIGNED_MAT}.matid'
            )
            filesystem.copyfile(default_id_file, os.path.join(comp_dir, os.path.basename(default_id_file)))
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                'material_default_display_options.json'
            )
            json_dict = read_display_options_from_json(default_file)
            categories.from_dict(json_dict)
            json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self.main_file))
            write_display_options_to_json(self.disp_opts_file, categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['display_uuid'] = categories.uuid
            self.data.commit()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            json_dict = duplicate_display_opts(new_path, os.path.basename(self.disp_opts_file))
            data = MaterialData(new_main_file)
            data.info.attrs['cov_uuid'] = ''
            data.info.attrs['display_uuid'] = json_dict['uuid']
            data.commit()

        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        action = ActionRequest(
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid

        initial_att_file = os.path.join(os.path.dirname(self.main_file), MAT_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), MAT_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            filesystem.removefile(initial_att_file)
            filesystem.removefile(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.polygon, att_id, comp_id)

        # Send the default material list to XMS.
        self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))
        return [], []

    def open_assign_poly_materials(self, query, params, win_cont):
        """Opens the Assign Materials dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        # Get the XMS attribute ids of the selected polygons (if any)
        _kwargs = params[0]
        poly_ids = _kwargs.get('selection', [])
        num_polys = len(poly_ids)
        if num_polys == 0:
            return [('INFO', 'No polygons selected. Select one or more polygons to assign materials.')], []

        # Get the component id (material id) map of the selected polygons (if any).
        current_mat_id = MaterialData.UNASSIGNED_MAT
        id_files = _kwargs.get('id_files', [])
        if id_files is not None and len(id_files) > 1:
            self.load_coverage_component_id_map({'POLYGON': (id_files[0], id_files[1])})
            # Delete the id dumped by xms files.
            if poly_ids:
                current_mat_id = self.get_comp_id(TargetType.polygon, poly_ids[0])  # First if multiple selected.
                if current_mat_id == -1 or current_mat_id is None:
                    current_mat_id = MaterialData.UNASSIGNED_MAT

        current_mat_idx = MaterialData.UNASSIGNED_MAT
        for idx, mat_id in enumerate(self.data.materials['id'].data):
            if mat_id == current_mat_id:
                current_mat_idx = idx
                break

        # Get original material ids, so we know if the user deletes one.
        old_ids = [int(x) for x in self.data.materials['id'].data.tolist()]

        title = 'Assign Material' if num_polys == 1 else 'Assign Materials'
        dlg = MaterialDialog(
            title, win_cont, self.main_file, self.is_sediment, query.display_projection.horizontal_units,
            current_mat_idx
        )
        if num_polys > 1:  # Add the multi-select warning if needed.
            dlg.add_multi_polygon_select_warning()

        if dlg.exec():
            self.update_material_list(query, old_ids)
            mat_id = int(self.data.materials['id'].data.tolist()[dlg.selected_material])
            for poly_id in poly_ids:
                self.update_component_id(TargetType.polygon, poly_id, mat_id)

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.main_file), 'temp'), ignore_errors=True)
        return [], []

    def update_material_list(self, query, old_ids):
        """Update the display options JSON file and display id files after editing the materials list.

        Args:
            query (:obj:`Query`): xmsapi interprocess communication object
            old_ids (:obj:`list[int]`): The old material ids
        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        # Check for materials removed from the list.
        self.data = MaterialData(self.main_file)
        new_ids = [int(x) for x in self.data.materials['id'].data.tolist()]
        deleted_ids = [int(x) for x in self.update_display_id_files(old_ids, new_ids)]
        self.unassign_materials(query, deleted_ids)
        # Write the display options file.
        self.update_display_options_file()
        self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))

    def unassign_materials(self, query, delete_ids):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            delete_ids (:obj:`list`): List of the deleted material ids.

        Returns:
            Empty message and ActionRequest lists

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        query.load_component_ids(self, polygons=True)
        if self.cov_uuid in self.comp_to_xms and TargetType.polygon in self.comp_to_xms[self.cov_uuid]:
            poly_map = self.comp_to_xms[self.cov_uuid][TargetType.polygon]
            for mat in delete_ids:
                if mat in poly_map:
                    for att_id in poly_map[mat]:
                        self.update_component_id(TargetType.polygon, att_id, MaterialData.UNASSIGNED_MAT)

    def update_display_options_file(self):
        """Update the XMS display options JSON file to match what we have in memory."""
        category_list = CategoryDisplayOptionList()
        category_list.target_type = TargetType.polygon
        category_list.uuid = str(self.data.info.attrs['display_uuid'])
        category_list.comp_uuid = self.uuid
        df = self.data.materials.to_dataframe()
        for mat_row in range(0, len(df.index)):
            category = CategoryDisplayOption()
            category.id = int(df.iloc[mat_row]['id'])
            category.description = df.iloc[mat_row]['Name']
            category.file = f'mat_{category.id}.matid'
            category.options = PolygonOptions()
            style = df.iloc[mat_row]['Color and Texture'].split()
            category.options.color = QColor(int(style[0]), int(style[1]), int(style[2]), 255)
            category.options.texture = int(style[3])
            # Make unassigned material the default category.
            if category.id == 0:  # Should always have 0 as "material id"
                category.is_unassigned_category = True
            category_list.categories.append(category)
        write_display_options_to_json(self.disp_opts_file, category_list)

    def open_material_properties(self, query, params, win_cont):
        """Opens the Material Properties dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        # get original material ids
        ids = [int(x) for x in self.data.materials['id'].data.tolist()]
        # dlg = MaterialParentDialog('Material List and Properties', win_cont, self.main_file, self.is_sediment,
        #                            query.display_projection.horizontal_units)
        dlg = MaterialDialog(
            'Material List and Properties', win_cont, self.main_file, self.is_sediment,
            query.display_projection.horizontal_units
        )
        if dlg.exec():
            self.update_material_list(query, ids)
        # from xms.srh.gui.mat_dialog import SrhMaterialDialog
        # dlg = SrhMaterialDialog('Material List and Properties', win_cont, self.main_file, self.is_sediment,
        #                         query.display_projection.horizontal_units)
        # dlg.exec()
        return [], []

    def update_display_id_files(self, old_ids, new_ids):
        """Update the display files.

        Args:
            old_ids (:obj:`list`): list of ids before editing materials
            new_ids (:obj:`list`): list of current material ids

        Returns:
            (:obj:`list`) : deleted ids
        """
        deleted_ids = old_ids
        path = os.path.dirname(self.main_file)
        for mat_id in new_ids:
            if mat_id >= 0:
                id_file = f'mat_{mat_id}.matid'
                filename = os.path.join(path, id_file)
                write_display_option_ids(filename, [mat_id])
            if mat_id in deleted_ids:
                deleted_ids.remove(mat_id)

        for mat_id in deleted_ids:
            id_file = f'mat_{mat_id}.matid'
            filename = os.path.join(path, id_file)
            filesystem.removefile(filename)
        return deleted_ids

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list[tuple]`):
                tuple containing:

                    main_file (:obj:`str`): The absolute path to the main file of the old component this
                    component is being merged from.

                    id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                    Each value is a tuple that may have two absolute file paths or none. The first
                    file is for the ids in XMS on the coverage. The second file contains the ids the
                    old component used for those objects. Both id files should be equal in length.
                    This dictionary is only applicable if the component derives from
                    CoverageComponentBase.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        main_file_idx = 0
        # att_file = 0
        # comp_file = 1
        # id_file_idx = 1
        # all_att_ids = []
        # new_comp_ids = []
        for m in merge_list:
            mat = MaterialData(m[main_file_idx])
            self.data.add_materials(mat)
            # old_to_new_mat_ids = self.data.add_materials(mat)
            # if len(m[id_file_idx]['POLYGON']) < 1:
            #     continue
            # att_ids = read_display_option_ids(m[id_file_idx]['POLYGON'][att_file])
            # comp_ids = read_display_option_ids(m[id_file_idx]['POLYGON'][comp_file])
            # all_att_ids.extend(att_ids)
            # for _, comp_id in zip(att_ids, comp_ids):
            #     new_comp_ids.append(int(old_to_new_mat_ids[comp_id]))
        # save attids and compids to a temp file so they can be processed later
        # tmpfname = os.path.join(os.path.dirname(self.main_file), 'tmp_att_ids_comp_ids.txt')
        # with open(tmpfname, 'w') as file:
        #     for att_id, comp_id in zip(all_att_ids, new_comp_ids):
        #         file.write(f'{att_id} {comp_id}\n')

        self.update_display_options_file()
        all_mat_ids = list(self.data.materials.to_dataframe()['id'].astype('i4'))
        self.update_display_id_files([], all_mat_ids)
        self.data.commit()

        action = ActionRequest(
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            comp_uuid=self.uuid,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def get_table_def(self, target_type):
        """Get the shapefile attribute table definition for a feature object.

        Args:
            target_type (:obj:`TargetType`): The feature object type

        Returns:
            (:obj:`list`): list of tuples where first element is column name and second element is ColAttType

        """
        if target_type == TargetType.polygon:
            return [
                ('ID', ColAttType.COL_ATT_INT), ('MATNAME', ColAttType.COL_ATT_STR), ('MATID', ColAttType.COL_ATT_INT),
                ('MANNINGS', ColAttType.COL_ATT_DBL), ('MATCOLOR', ColAttType.COL_ATT_STR)
            ]
        return None

    def get_att_table(self, target_type):
        """Get the shapefile attribute table for a feature object.

        Args:
            target_type (:obj:`TargetType`): The feature object type

        Returns:
            (:obj:`pandas.DataFrame`): The shapefile attributes to write

        """
        if target_type == TargetType.polygon:
            data = {'ID': [], 'MATNAME': [], 'MATID': [], 'MANNINGS': [], 'MATCOLOR': []}
            df = self.data.materials.to_dataframe()
            cov_dict = self.comp_to_xms.get(self.cov_uuid, {})
            poly_dict = cov_dict.get(TargetType.polygon, {})
            for comp_id, xms_id_list in poly_dict.items():
                record = df.loc[df['id'] == comp_id].reset_index(drop=True).to_dict()
                mat_name = record['Name'].get(0, 'unassigned')
                mat_color = record['Color and Texture'].get(0, '0 0 0 1')
                manning = record["Manning's N"].get(0, 0.02)
                for xms_id in xms_id_list:
                    data['ID'].append(xms_id)
                    data['MATNAME'].append(mat_name)
                    data['MATID'].append(comp_id)
                    data['MANNINGS'].append(manning)
                    data['MATCOLOR'].append(mat_color)
            return pd.DataFrame(data)
        return None

    def populate_from_att_tables(self, att_dfs):
        """Populate attributes from an attribute table written by XMS.

        Args:
            att_dfs (:obj:`dict`): Dictionary of attribute pandas.DataFrame. Key is TargetType enum,
                value is None if not applicable.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        df = att_dfs[TargetType.polygon]
        # xms_ids = df['ID'].astype(int)
        xms_ids = df['ID'].to_list()
        mat_names = [x if pd.notnull(x) else '' for x in df['MATNAME'].to_list()]
        df = df.drop_duplicates(['MATNAME'])
        df = df.fillna(value={'MATNAME': '', 'MATCOLOR': '0 0 0 1', 'MANNINGS': 0.02})
        if len(df) == 1 and not df.loc[0, 'MATNAME']:  # Ensure material name field was mapped
            msg = 'MATNAME must be mapped to shapefile attribute to import material properties.'
            return [('INFO', msg)], []
        # set default color and pattern
        next_color = 0
        next_pattern = PolygonTexture.cross_pattern
        send_back_mat_color_error = False
        for idx, row in df.iterrows():
            # make sure color is formatted correctly
            mat_color_str_error = False
            words = row['MATCOLOR'].split()
            if len(words) != 4:
                mat_color_str_error = True
            else:
                for i, word in enumerate(words):
                    try:
                        val = int(word)
                        if i < 3 and (val < 0 or val > 255):
                            mat_color_str_error = True
                        elif i == 3 and (val < 0 or val > PolygonTexture.null_pattern):
                            mat_color_str_error = True
                    except ValueError:
                        mat_color_str_error = True
            if mat_color_str_error:
                row['MATCOLOR'] = '0 0 0 1'
                send_back_mat_color_error = True

            # default color and pattern
            if row['MATNAME'] and row['MATCOLOR'] == '0 0 0 1':
                if next_color >= len(DEFAULT_MATERIAL_COLORS):
                    next_color = 0
                if next_pattern > PolygonTexture.null_pattern:
                    next_pattern = PolygonTexture.cross_pattern
                clr = DEFAULT_MATERIAL_COLORS[next_color]
                clr_str = f'{clr[0]} {clr[1]} {clr[2]} {int(next_pattern)}'
                df.at[idx, 'MATCOLOR'] = clr_str
                next_color += 1
                next_pattern += 1
        df.index = pd.RangeIndex(len(df.index))

        temp_folder = os.path.join(os.getcwd(), str(uuid.uuid4()))
        os.makedirs(temp_folder)
        mainfile = os.path.join(temp_folder, str(uuid.uuid4()))
        mat_comp = MaterialComponent(mainfile)
        mat_df = mat_comp.data.materials.to_dataframe()
        mat_name_comp_id = {'': 0}
        mat_id = max(self.data.materials.to_dataframe()['id'].to_list())
        for i in range(len(df)):
            mat_name = df.loc[i, 'MATNAME']
            if not mat_name:
                continue
            row = len(mat_df)
            mat_id += 1
            mat_name_comp_id[mat_name] = mat_id
            mat_df.loc[row, 'id'] = mat_id
            mat_df.loc[row, 'Name'] = mat_name
            mat_df.loc[row, "Manning's N"] = df.loc[i, 'MANNINGS']
            mat_df.loc[row, 'Color and Texture'] = df.loc[i, 'MATCOLOR']
            mat_df.loc[row, 'Depth Varied Curve'] = 0
        mat_df['id'] = mat_df['id'].astype(int)
        mat_df['Depth Varied Curve'] = mat_df['Depth Varied Curve'].astype(int)
        mat_comp.data.set_materials(mat_df.to_xarray())
        mat_comp.data.commit()

        comp_ids = [mat_name_comp_id[mat_name] for mat_name in mat_names]
        for att_id, comp_id in zip(xms_ids, comp_ids):
            self.update_component_id(TargetType.polygon, att_id, comp_id)
        id_file = os.path.join(temp_folder, str(uuid.uuid4()))
        comp_id_file = os.path.join(temp_folder, str(uuid.uuid4()))
        write_display_option_ids(id_file, xms_ids)
        write_display_option_ids(comp_id_file, comp_ids)
        shapefile_comp = [(mainfile, {'POLYGON': (id_file, comp_id_file)})]
        msgs, actions = self.handle_merge(shapefile_comp)

        mat_comp.data.close()
        shutil.rmtree(temp_folder, ignore_errors=True)
        if send_back_mat_color_error:
            msgs.append(('WARNING', 'Invalid format for material color. Default color and pattern assigned.'))
        return msgs, actions
