"""Exporter for SRH-2D PEST runs."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.srh.components.parameters.parameters_manager import ParametersManager

# 4. Local modules


class PestPstWriter:
    """Write files needed for an SRH-2D PEST run."""
    def __init__(self, file_name, data, main_file=''):
        """Constructor.

        Args:
            file_name (:obj:`str`): Path to the file to export
            data (:obj:`dict`): Json dict of PEST run options
            main_file (:obj:`str`): Path to the simulation component's main file
        """
        self._file_name = file_name
        self._file = None
        self._data = data
        self._param_data = ParametersManager.read_parameter_file(main_file)
        self._npar = 0
        self._noptmax = 10  # from user
        self._nphistp = 2  # from user
        if self._param_data:
            self._noptmax = self._param_data['max_iterations']
            self._nphistp = self._param_data['max_iterations_no_improvement']
            for row, _id in enumerate(self._param_data['params']['id']):
                if self._is_inlet_q_or_manning(_id):
                    if self._param_data['params']['Use'][row]:
                        self._npar += 1
                        plot_name = f"{self._param_data['params']['Type'][row]} - "
                        plot_name += f"{self._param_data['params']['Description'][row]}"
                        self._data['PARAMETER_NAMES'].append(plot_name)
        # need to calculate this
        self._nobs = len(self._data['OBSERVATIONS'])

    def _is_inlet_q_or_manning(self, _id):
        """Returns True if id indicates Inlet Q or Manning's N.

        Args:
            _id (:obj:`str`): The ID string.

        Returns:
            (:obj:`bool`): See description.
        """
        return _id.startswith('inlet_q') or _id.startswith('mat')

    def write(self):
        """Write the PEST input files."""
        with open(self._file_name, 'w') as self._file:
            self._file.write('pcf\n')
            self._write_control_data()
            self._write_svd_data()
            self._write_parameter_groups()
            self._write_parameter_data()
            self._write_obs_groups()
            self._write_obs_data()
            self._write_model_command()
            self._write_model_io(os.path.splitext(os.path.basename(self._file_name))[0])
            self._write_prior_info()
            self._write_regularization_data()

    def _write_control_data(self):
        """Write the PEST control data to the PST file."""
        self._file.write('* control data\n')
        self._file.write('restart estimation\n')  # RSTFLE PESTMODE
        # NPAR NOBS NPARGP NPRIOR NOBSGP [MAXCOMPDIM] [DERZEROLIM]
        self._file.write(f'{self._npar} {self._nobs} 1 0 1\n')
        # NTPLFLE NINSFLE PRECIS DPOINT [NUMCOM JACFILE MESSFILE] [OBSREREF]
        self._file.write('1 1 single point 1 0 0\n')
        # RLAMBDA1 RLAMFAC PHIRATSUF PHIREDLAM NUMLAM [JACUPDATE] [LAMFORGIVE] [DERFORGIVE]
        self._file.write('10.0 2.0 0.3 0.03 10\n')
        # RELPARMAX FACPARMAX FACORIG [IBOUNDSTICK] [UPVECBEND] [ABSPARMAX]
        self._file.write('5.0 5.0 0.001\n')
        # PHIREDSWH [NOPTSWITCH] [SPLITSWH] [DOAUI] [DOSENREUSE] [BOUNDSCALE]
        self._file.write('0.1\n')
        # [AUI] don't write this line
        # NOPTMAX PHIREDSTP NPHISTP NPHINORED RELPARSTP NRELPAR [PHISTOPTHRESH] [LASTRUN] [PHIABANDON]
        self._file.write(f'{self._noptmax} 0.005 {self._nphistp} 3 0.005 3\n')
        # ICOV ICOR IEIG [IRES] [JCOSAVE] [VERBOSEREC] [JCOSAVEITN] [REISAVEITN] [PARSAVEITN]
        self._file.write('1 1 1\n')
        # [PARSAVERUN] don't write this line

    def _write_svd_data(self):
        """Write SVD options to PST file."""
        self._file.write('* singular value decomposition\n')
        # SVDMODE
        self._file.write('1\n')
        # MAXSING EIGTHRESH
        self._file.write('1000 1.0e-007\n')
        # EIGWRITE
        self._file.write('0\n')

    def _write_parameter_groups(self):
        """Write parameter groups section to PST file."""
        self._file.write('* parameter groups\n')
        # PARGPNME INCTYP DERINC DERINCLB FORCEN DERINCMUL DERMTHD
        # (one such line for each of the NPARGP parameter groups)
        self._file.write('general relative 0.01 0.0 switch 2.0 outside_pts\n')

    def _write_parameter_data(self):
        """Write parameter data to the PST file."""
        self._file.write('* parameter data\n')
        if not self._param_data:
            return  # pragma no cover
        # PARNME PARTRANS PARCHGLIM PARVAL1 PARLBND PARUBND PARGP SCALE OFFSET DERCOM
        # (one such line for each of the NPAR parameters)
        for row, _id in enumerate(self._param_data['params']['id']):
            if self._is_inlet_q_or_manning(_id):
                if self._param_data['params']['Use'][row]:
                    name = self._param_data['params']['id'][row]
                    value = self._param_data['params']['Value'][row]
                    _min = self._param_data['params']['Min'][row]
                    _max = self._param_data['params']['Max'][row]
                    self._file.write(f"{'{0: <12}'.format(name)} none   factor        ")
                    self._file.write(f"{'{0: <12}'.format(value)} ")
                    self._file.write(f"{'{0: <13}'.format(_min)} ")
                    self._file.write(f"{'{0: <12}'.format(_max)} ")
                    self._file.write("general        1.000       0.000        1\n")
        # PARNME PARTIED
        # (one such line for each tied parameter)

    def _write_obs_groups(self):
        """Write observation groups to PST file."""
        self._file.write('* observation groups\n')
        # OBGNME (one such line for each observation group) 12 characters or less
        self._file.write('WSE\n')

    def _write_obs_data(self):
        """Write observation data to PST file."""
        self._file.write('* observation data\n')
        # OBSNME OBSVAL WEIGHT OBGNME
        # (one such line for each of the NOBS observations)
        obs = self._data['OBSERVATIONS']
        num_obs = len(obs)
        for i in range(1, num_obs + 1):
            ob_val = f"{'{0: < 13}'.format(obs[i-1][0])}"
            ob_wt = f"{'{0: < 13}'.format(obs[i-1][1])}"
            self._file.write(f"wse{i} {ob_val} {ob_wt} WSE\n")

    def _write_model_command(self):
        """Write the model command data to PST file."""
        self._file.write('* model command line\n')
        self._file.write('start /wait /min pest_run_model.bat\n')

    def _write_model_io(self, base_name):
        """Write the model io to the PST file."""
        self._file.write('* model input/output\n')
        self._file.write(f'"{base_name}.srhhydro.tpl" "{base_name}.srhhydro"\n')
        self._file.write(f'"{base_name}_obs.ins" "{base_name}_obs_out.txt"\n')

    def _write_prior_info(self):
        """Write prior information to PST file."""
        # not needed for SRH, maybe do preferred value because of the lack of
        # observation data
        pass

    def _write_regularization_data(self):
        """Write regularization data to PST file."""
        # not needed for SRH
        pass
