"""Retrieve data needed by the Manning's N dialog from XMS."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import shutil
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Component, Coverage, Simulation
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.srh.components.bc_component import BC_INITIAL_ATT_ID_FILE
from xms.srh.components.bc_component import BC_INITIAL_COMP_ID_FILE
from xms.srh.components.bc_component import BcComponent


class CreateFloodwaySimRunner(ProcessFeedbackThread):
    """Worker thread for creating a floodway simulation."""
    def __init__(self, query, parent, sim_component, wse_maximum_rise, mat_uuid):
        """Constructor.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
            parent (:obj:`QWidget`): Parent Qt dialog
            sim_component (:obj:`SimComponent`): The simulation's component
            wse_maximum_rise (:obj:`dict`): Maximum WSE rise variable
            mat_uuid (:obj:`str`): UUID of the Materials coverage to link to the simulation
        """
        super().__init__(do_work=self._do_work, parent=parent)
        self._logger = logging.getLogger('xms.srh')
        self.query = query
        self.sim_component = sim_component
        self.wse_maximum_rise = wse_maximum_rise
        self.mat_uuid = mat_uuid
        self.new_main_file = ""
        self.old_sim = None
        self.new_sim = None
        self.new_sim_comp = None
        self.new_bc_cov = None
        self.new_bc_comp = None
        self.bc_comp_keywords = None

    def _do_work(self):
        """Creates a floodway simulation from the current simulation."""
        try:
            # Duplicate the simulation
            self._create_simulation_copy()

            # Get the old sim's tree item
            old_sim_node = tree_util.find_tree_node_by_uuid(self.query.project_tree, self.old_sim.uuid)

            # Duplicate the BC coverage
            self._duplicate_bc_coverage(old_sim_node)

            # Add new simulation, new BC coverage, and link items to the new simulation
            self._add_built_data(old_sim_node)
        except Exception as e:
            message = str(e)
            if len(message) == 0:
                message = 'Error creating floodway simulation.'
            self._logger.exception(message)
        finally:
            self.processing_finished.emit()

    def _create_simulation_copy(self):
        """Duplicate an SRH-2D simulation and its component data."""
        try:
            # Get the old simulation's information
            self.old_sim = self.query.parent_item()
            new_sim_name = f'{self.old_sim.name} (Floodway)'
            self.new_sim = Simulation(sim_uuid=str(uuid.uuid4()), model='SRH-2D', name=new_sim_name)

            # Create a copy of the old simulation's hidden component
            new_comp_uuid = str(uuid.uuid4())
            new_comp_path = os.path.join(os.path.dirname(os.path.dirname(self.sim_component.main_file)), new_comp_uuid)
            src = os.path.dirname(self.sim_component.main_file)
            shutil.copytree(src, new_comp_path)
            self.new_main_file, _, _ = self.sim_component.save_to_location(new_comp_path, 'DUPLICATE')
            # Create the data_objects component
            self.new_sim_comp = Component(
                main_file=self.new_main_file, comp_uuid=new_comp_uuid, unique_name='Sim_Manager', model_name='SRH-2D'
            )
        except Exception:
            raise Exception('Unable to duplicate simulation.')

    def _duplicate_bc_coverage(self, old_sim_node):
        """Duplicate an SRH-2D Boundary Conditions coverage and its component data.

        Args:
            old_sim_node (:obj:`TreeNode`): Parent simulation of the BC coverage to duplicate
        """
        try:
            # Should be one and only one linked BC coverage.
            old_bc_nodes = tree_util.descendants_of_type(
                old_sim_node, xms_types=['TI_COVER_PTR'], allow_pointers=True, coverage_type='Boundary Conditions'
            )
            if not old_bc_nodes:
                raise RuntimeError()

            # Get the old BC coverage geometry and hidden component main file
            old_bc_cov = self.query.item_with_uuid(old_bc_nodes[0].uuid)
            old_bc_do_comp = self.query.item_with_uuid(
                old_bc_nodes[0].uuid, model_name='SRH-2D', unique_name='Bc_Component'
            )

            new_cov_uuid = str(uuid.uuid4())
            new_cov_name = f'{old_bc_cov.name} (Floodway)'
            self.new_bc_cov = Coverage(name=new_cov_name, uuid=new_cov_uuid)
            old_bc_arcs = old_bc_cov.arcs
            self.new_bc_cov.arcs = old_bc_arcs
            self.new_bc_cov.complete()

            # Create a copy of the old BC coverage's hidden component
            new_comp_uuid = str(uuid.uuid4())
            new_comp_path = os.path.join(os.path.dirname(os.path.dirname(old_bc_do_comp.main_file)), new_comp_uuid)
            old_bc_py_comp = BcComponent(old_bc_do_comp.main_file)
            src = os.path.dirname(old_bc_do_comp.main_file)
            shutil.copytree(src, new_comp_path)
            self.new_main_file, _, _ = old_bc_py_comp.save_to_location(new_comp_path, 'DUPLICATE')

            # Copy the old component id mappings
            files_dict = self.query.load_component_ids(old_bc_py_comp, arcs=True, delete_files=False)
            att_id_file = files_dict['ARC'][0]
            comp_id_file = files_dict['ARC'][1]
            filesystem.copyfile(att_id_file, os.path.join(new_comp_path, BC_INITIAL_ATT_ID_FILE))
            filesystem.copyfile(comp_id_file, os.path.join(new_comp_path, BC_INITIAL_COMP_ID_FILE))

            # Create the data_objects component
            self.new_bc_comp = Component(
                main_file=self.new_main_file, model_name='SRH-2D', unique_name='Bc_Component', comp_uuid=new_comp_uuid
            )

            # Update the python component data
            new_bc_py_comp = BcComponent(self.new_main_file)
            new_bc_py_comp.cov_uuid = new_cov_uuid
            new_bc_py_comp.data.info.attrs['cov_uuid'] = new_cov_uuid
            new_bc_py_comp.do_init_disp_opts()
            new_bc_py_comp.load_coverage_component_id_map(files_dict)  # Load the component ids
            filesystem.removefile(att_id_file)  # Clean up the files written by XMS
            filesystem.removefile(comp_id_file)

            for arc in old_bc_arcs:
                comp_id = new_bc_py_comp.get_comp_id(TargetType.arc, arc.id)
                bc_id = new_bc_py_comp.data.bc_id_from_comp_id(comp_id)
                bc_data_param = new_bc_py_comp.data.bc_data_param_from_id(bc_id)
                if bc_data_param.bc_type == 'Exit-H (subcritical outflow)':
                    exit_h = bc_data_param.exit_h
                    if exit_h.water_surface_elevation_option == 'Constant':
                        units = exit_h.constant_wse_units
                        rise = self.wse_maximum_rise[units]
                        exit_h.constant_wse += rise
                    elif exit_h.water_surface_elevation_option == 'Time series':
                        df = exit_h.time_series_wse
                        if exit_h.time_series_wse_units == 'hrs -vs- feet':
                            units = 'Feet'
                        else:
                            units = 'Meters'
                        rise = self.wse_maximum_rise[units]
                        df['y'] = df['y'].apply(lambda x, rise_val=rise: x + rise_val)
                    elif exit_h.water_surface_elevation_option == 'Rating curve':
                        self._logger.warning(
                            "Unable to update boundary conditions for a specified rating curve. Please"
                            " select a different exit boundary condition."
                        )
                    new_bc_py_comp.data.set_bc_data_with_id(bc_data_param, bc_id)
            new_bc_py_comp.display_option_list.append(
                XmsDisplayMessage(file=new_bc_py_comp.disp_opts_file, edit_uuid=new_bc_py_comp.cov_uuid)
            )
            new_bc_py_comp.data.commit()
            self.bc_comp_keywords = [
                {
                    'component_coverage_ids': [new_bc_py_comp.uuid, new_bc_py_comp.update_ids],
                    'display_options': new_bc_py_comp.get_display_options()
                }
            ]
        except Exception:
            raise Exception('Unable to duplicate Boundary Condition coverage.')

    def _add_built_data(self, old_sim_node):
        """Add the new simulation and BC coverage to XMS.

        Args:
            old_sim_node (:obj:`TreeNode`): The simulation project explorer item
        """
        # Add the new simulation.
        self.query.add_simulation(self.new_sim, components=[self.new_sim_comp])
        self.query.add_coverage(
            self.new_bc_cov,
            model_name='SRH-2D',
            coverage_type='Boundary Conditions',
            components=[self.new_bc_comp],
            component_keywords=self.bc_comp_keywords
        )

        # Link the new BC coverage to the new simulation.
        self.query.link_item(taker_uuid=self.new_sim.uuid, taken_uuid=self.new_bc_cov.uuid)

        # Link the Materials coverage if it was specified
        if self.mat_uuid:
            self.query.link_item(taker_uuid=self.new_sim.uuid, taken_uuid=self.mat_uuid)

        # Copy links for the mesh and any Monitor or Obstructions coverages to the new simulation.
        snapped_components = [
            'Mapped_Material_Component', 'Mapped_Sed_Material_Component', 'Mapped_Bc_Component',
            'Mapped_Monitor_Component'
        ]
        for child in old_sim_node.children:
            if child.unique_name in snapped_components:
                continue
            is_skip_coverage = child.coverage_type in ['Materials', 'Sediment Materials', 'Boundary Conditions']
            is_solution = child.item_typename == 'TI_SOLUTION_FOLDER'
            if not is_skip_coverage and not is_solution:
                self.query.link_item(taker_uuid=self.new_sim.uuid, taken_uuid=child.uuid)


def create_floodway_simulation_with_feedback(query, parent, sim_component, wse_maximum_rise, mat_uuid):
    """Run the create floodway simulation tool with a feedback dialog.

    Args:
        query (:obj:`Query`): XMS interprocess communication object
        parent (:obj:`QWidget`): Parent Qt dialog
        sim_component (:obj:`SimComponent`): The simulation's component
        wse_maximum_rise (:obj:`dict`): Maximum WSE rise variable
        mat_uuid (:obj:`str`): UUID of the Materials coverage to link to the simulation
    """
    worker = CreateFloodwaySimRunner(query, parent, sim_component, wse_maximum_rise, mat_uuid)
    display_text = {
        'title': 'Generate SRH-2D Floodway Simulation',
        'working_prompt': 'Generating SRH-2D floodway simulation. Please wait...',
        'warning_prompt':
            'Warning(s) encountered generating the SRH-2D floodway simulation. Review log '
            'output for more details.',
        'error_prompt':
            'Error(s) encountered generating the SRH-2D floodway simulation. Review log output for'
            ' more details.',
        'success_prompt': 'Successfully generated new SRH-2D floodway simulation with necessary updated'
                          ' coverages',
        'note': '',
        'auto_load': 'Auto load data upon successful completion',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srh', worker, parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    feedback_dlg.exec()
