"""Migrate old DMI obstruction coverage attributes."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.data.obstruction_data import ObstructionData
from xms.srh.data.par.obstruction_param import ObstructionParam


class MigrateObstructionAtts:
    """Convert attributes on Obstructions coverage arcs/sets to component-based data."""
    def __init__(self, cov_points, pt_widget_map, cov_arcs, arc_widget_map):
        """Construct the migrator.

        Args:
            cov_points (:obj:`list`): List of the xms.data_objects.Parameters.Point objects in the obstructions
                coverage.
            pt_widget_map (:obj:`dict`): Dict of point database widget values for a particular coverage that we want to
                fill.
            cov_arcs (:obj:`list`): List of the xms.data_objects.Parameters.Arc objects in the obstructions coverage.
            arc_widget_map (:obj:`dict`): Dict of arc database widget values for a particular coverage that we want to
                fill.
        """
        self._pts = cov_points
        self._arcs = cov_arcs
        self._arc_widget_map = arc_widget_map
        self._pt_widget_map = pt_widget_map
        self._ids = []
        self._obs = []
        self.pier_ids = {}
        self.deck_ids = {}
        self.obs_data = None

    def migrate(self):
        """Migrate the obstructions point/arc attributes."""
        comp_id = 1  # Start at 1 so the first one isn't treated as default unassigned.
        for obs_pt in self._pts:  # Add disjoint pier points first if we have any
            att_id = obs_pt.id
            self.pier_ids[att_id] = comp_id
            self._ids.append(comp_id)
            self._fill_obs_atts_from_db(self._pt_widget_map, att_id)
            comp_id += 1
        for obs_arc in self._arcs:  # Add deck arcs if we have any
            att_id = obs_arc.id
            self.deck_ids[att_id] = comp_id
            self._ids.append(comp_id)
            self._fill_obs_atts_from_db(self._arc_widget_map, att_id)
            comp_id += 1
        import tempfile
        with tempfile.NamedTemporaryFile(mode='wt') as temp:
            filename = temp.name
        obs_data = ObstructionData(filename)
        for i in range(len(self._ids)):
            obs_data.append_obstruction_data_with_id(self._obs[i], self._ids[i])
        self.obs_data = obs_data.obstruction_data

    def _fill_obs_atts_from_db(self, db_atts, att_id):
        """Get attributes of an obstruction point or arc from database dict.

        Args:
            db_atts (:obj:`dict`): Dict of database widget values for the obstruction points or arcs in the format
                passed in by xms.components run_migrate
            att_id (:obj:`int`): Feature att id of the obstruction point or arc
        """
        self._obs.append(ObstructionParam())
        atts = db_atts.get(att_id)
        if atts:
            self._obs[-1].width = atts['edtObstructionWidth'][0][2]
            self._obs[-1].thickness = atts['edtObstructionThickness'][0][2]
            self._obs[-1].drag = atts['edtObstructionDragCoef'][0][2]
            self._obs[-1].units = 'Feet' if atts['cbxObstructionUnits'][0][2] == 'ft' else 'Meters'
            self._obs[-1].porosity = atts['edtObstructionPorosity'][0][2]
