"""Script to launch srh pest post."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import argparse
import csv
import datetime
import json
import os
import subprocess
import sys
import time

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.srh.model.srh_ascii_art import HY8_TXT, POST_TXT, PRE_TXT, SRH_TXT
from xms.srh.model.srh_post_runner import run_post_process


class ModelRunner:
    """Class for running SRH model."""
    def __init__(self):
        """Constructor."""
        self.args = None
        self.start_dir = os.getcwd()
        self.run_dir = ''
        self.diagnostic = ''
        self.exit_code = 0
        self.cmd = None
        self.stdout = None
        self.log_file = ''
        self._run_file_hy8 = ''
        self._pid_hy8 = -1

    def parse_args(self):
        """Parses command line arguments."""
        parser = argparse.ArgumentParser(description="SRH scenario runner.")
        parser.add_argument('-j', '--json_file', help='JSON file with model run info', required=False, default='')
        self.args = parser.parse_args()

    def _run_cmd(self):
        """Runs a command line and looks for a string that indicates success."""
        popen_obj = subprocess.Popen(self.cmd, stdout=self.stdout, stderr=self.stdout)
        # child process id so XMS can kill it on abort
        child_proc_file = f'{os.getpid()}_child_proc.txt'
        fname = os.path.join(XmEnv.xms_environ_temp_directory(), child_proc_file)
        with open(fname, 'w') as f:
            f.write(f'{popen_obj.pid}\n')

        if self._run_file_hy8:  # hy8 process id so other running scripts know to wait
            with open(self._run_file_hy8, 'w') as f:
                f.write(f'Process id:\n{popen_obj.pid}\n')

        popen_obj.wait()
        # check log for success string
        if not self.success_str:
            return

        with open(self.log_file, 'r') as f:
            lines = f.readlines()
            ll = [line for line in lines if line.find(self.success_str) != -1]
            if not ll:
                self.exit_code = -1
                raise RuntimeError

    def _print_message_with_date_time(self, msg):
        """Print a message and attach the date time.

        Args:
            msg (:obj:`str`): the message
        """
        now_str = datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        self.stdout.write(f'\n{msg} {now_str}\n\n')

    def _hy8_is_running(self):
        """Check if hy-8 is running.

        Returns:
            (:obj:`bool`): True if the process is running
        """
        if self._pid_hy8 < 0:
            return False

        p_list = subprocess.Popen('tasklist.exe /fo csv', stdout=subprocess.PIPE, universal_newlines=True)
        set_pid = {int(p['PID']) for p in csv.DictReader(p_list.stdout)}
        if self._pid_hy8 in set_pid:
            return True
        return False

    def _pid_from_hy8_run_file(self):
        """Gets the process id from the hy-8 run file."""
        def read_pid():
            with open(self._run_file_hy8, 'r') as f:
                lines = f.readlines()
                if len(lines) == 2:
                    self._pid_hy8 = int(lines[1])
                    self.stdout.write(f'HY-8 process id: {self._pid_hy8}\n')

        if os.path.isfile(self._run_file_hy8):
            self.stdout.write('Getting HY-8 process id.\n')
            read_pid()
            if self._pid_hy8 < 0:
                time.sleep(2)
                read_pid()
            self.stdout.flush()

    def _check_if_hy8_needs_to_run(self):
        """Checks if HY-8 needs to run."""
        self._run_file_hy8 = os.path.join(XmEnv.xms_environ_temp_directory(), 'hy8_running.txt')
        self.stdout.write('Checking if HY-8 is running\n')
        if os.path.isfile(self._run_file_hy8):
            time.sleep(2)
        self._pid_from_hy8_run_file()

        if self._hy8_is_running():
            self.stdout.write('HY-8 must run before running SRH.\n')
            while self._hy8_is_running():
                self.stdout.write('WAITING FOR HY-8 TO FINISH RUNNING\n')
                self.stdout.flush()
                time.sleep(3)

        # if the table file is newer than the hy8 input file then don't run hy8
        hy8_file = self.json_data['hy8_input']
        tab_file = hy8_file.replace('hy8', 'table')
        if os.path.isfile(tab_file) and os.path.getsize(tab_file) > 0:
            if os.path.getmtime(hy8_file) < os.path.getmtime(tab_file):
                self.stdout.write('HY-8 output file is newer than HY-8 input file. No need to run HY-8.\n')
                self.stdout.flush()
                self._run_file_hy8 = ''
                return

    def _run_hy8_from_json(self):
        """Run HY-8 from the json input."""
        if 'hy8_exe' not in self.json_data:
            return
        self._check_if_hy8_needs_to_run()
        if not self._run_file_hy8:
            return

        self.stdout.write('HY-8 must run before running SRH.\n')
        cmds = [
            self.json_data['hy8_exe'], '-buildFlowTwTable', 'FLOWCOEF', '1.1', 'FLOWCONST', '0.25', 'UNITS', 'EN',
            'HWINC', '0.25', 'TWINC', '0.25', 'MAXRECURSION', '15', 'MAXITERATION', '15', self.json_data['hy8_input']
        ]
        self.cmd = cmds
        self.success_str = ''
        self.stdout.write(HY8_TXT)
        self._print_message_with_date_time('HY-8 started at ')
        self.stdout.flush()
        self._run_cmd()

        if os.path.isfile(self._run_file_hy8):
            os.remove(self._run_file_hy8)

        tab_file = self.json_data['hy8_input'].replace('hy8', 'table')
        if not os.path.isfile(tab_file):
            self.exit_code = -3
            raise RuntimeError
        if os.path.getsize(tab_file) < 1:
            self.exit_code = -4
            raise RuntimeError
        self.stdout.write('HY-8 terminated successfully\n\n')
        self._print_message_with_date_time('HY-8 finished at ')
        self.stdout.flush()
        self._run_file_hy8 = ''

    def _run_srh_pre_from_json(self):
        """Run SRH-PRE from the json input."""
        in_file = self.json_data['srh_pre_input']
        dir_name = os.path.dirname(in_file)
        self.run_dir = os.path.normpath(os.path.join(self.start_dir, dir_name))
        in_file = os.path.basename(in_file)
        os.chdir(self.run_dir)
        if in_file.lower().find('.srhhydro') != -1:
            inputs = ['3', in_file]
        else:
            inputs = [in_file]

        self.stdout.write(PRE_TXT)
        self._print_message_with_date_time('SRH-PRE started at ')
        self.stdout.flush()

        self.cmd = [self.json_data['srh_pre_exe']] + inputs
        self.success_str = 'SRH Preprocessor is successfully executed'
        self._run_cmd()

        self._print_message_with_date_time('SRH-PRE finished at ')
        self.stdout.flush()

    def _run_srh_from_json(self):
        """Run SRH-2d from the json input."""
        in_file = self.json_data['srh_exe_input']
        dir_name = os.path.dirname(in_file)
        self.run_dir = os.path.normpath(os.path.join(self.start_dir, dir_name))
        in_file = os.path.basename(in_file)
        os.chdir(self.run_dir)
        self.cmd = [self.json_data['srh_exe'], in_file]
        self.success_str = 'SRH is successfully executed'

        self.stdout.write(SRH_TXT)
        self._print_message_with_date_time('SRH-2D started at ')
        self.stdout.flush()
        self._run_cmd()

        self._print_message_with_date_time('SRH-2D finished at ')
        self.stdout.flush()

    def _run_post_from_json(self):
        """Run SRH post from the json input."""
        if 'srh_post_input' not in self.json_data:
            return

        self.stdout.write('\n\nSRH POST RUNNING\n\n')
        from contextlib import redirect_stdout
        with redirect_stdout(self.stdout):
            self.stdout.write(POST_TXT)
            self._print_message_with_date_time('SRH-POST started at ')
            self.stdout.flush()
            run_post_process(self.json_data['srh_post_input'])
            self._print_message_with_date_time('SRH-POST finished at ')
            self.stdout.flush()

    def _run_from_json(self):
        """Runs model from info in json file."""
        self.args.json_file = self.args.json_file.strip()
        with open(self.args.json_file, 'r') as f:
            self.json_data = json.load(f)

        # check for necessary data in json file
        required = ['srh_exe', 'srh_exe_input', 'srh_pre_exe', 'srh_pre_input']
        if not all(k in self.json_data for k in required):
            self.exit_code = 2
            return

        self.log_file = XmEnv.xms_environ_stdout_file()
        with open(self.log_file, 'a') as self.stdout:
            self._run_hy8_from_json()
            self._run_srh_pre_from_json()
            self._run_srh_from_json()
            self._run_post_from_json()

    def run(self):
        """Runs the model."""
        try:
            self._run_from_json()
        except RuntimeError:
            pass


if __name__ == '__main__':
    runner = ModelRunner()
    runner.parse_args()
    runner.run()
    sys.exit(runner.exit_code)
