"""DifferenceViewer class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import partial
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QDialogButtonBox, QLabel, QPushButton, QTableWidget, QTableWidgetItem, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.srh_swmm_coupler.gui.dataframe_viewer import DataframeViewer


class DifferenceViewer(XmsDlg):
    """The DifferenceViewer Dialog."""

    COL_SINK_LABEL = 0
    """The column in the table for the inlet name"""
    COL_MAX_DELTA = 1
    """The column in the table for the max delta values"""
    COL_TIME_SERIES = 2
    """The column in the table for the time series buttons"""

    def __init__(self, dataframes: dict, title: str, parameter_string: str, parent=None):
        """Initializes the dialog.

        Args:
            dataframes (dict of pd.DataFrame): The dict of Pandas Dataframes, with the sink labels as keys.
            title (str): The dialog title.
            parameter_string: The parameter string to use for the output windows.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.srh_swmm_coupler.gui.difference_viewer')

        self._dataframes = dataframes
        self.setWindowTitle(title)
        self._parameter_string = parameter_string
        vertical_layout = QVBoxLayout(self)
        self._create_table_widget(vertical_layout)
        self._add_items()
        self.table_widget.resizeColumnsToContents()
        self._add_max_delta(vertical_layout)
        self._create_button_box(vertical_layout)

    def _create_table_widget(self, vertical_layout: QVBoxLayout):
        """Creates the table widget.

        Args:
            vertical_layout: The layout.
        """
        column_count = self.COL_TIME_SERIES + 1
        self.table_widget = QTableWidget(len(self._dataframes), column_count, self)
        column_names = [f'{self._parameter_string} Name', 'Maximum Change', 'Details']
        self.table_widget.setHorizontalHeaderLabels(column_names)
        vertical_layout.addWidget(self.table_widget)

    def _add_items(self):
        """Adds the list to the rows of the table."""
        for row, (sink_label, value) in enumerate(self._dataframes.items()):
            self._add_items_to_row(row, sink_label, value)

    def _create_button_box(self, vertical_layout: QVBoxLayout):
        """Creates the button box.

        Args:
            vertical_layout: The layout
        """
        self.btn_box = QDialogButtonBox()
        self.btn_box.setStandardButtons(QDialogButtonBox.Close)
        self.btn_box.rejected.connect(self.reject)
        vertical_layout.addWidget(self.btn_box)

    def _add_items_to_row(self, row: int, sink_label: str, dataframe: pd.DataFrame):
        """Adds the widgets to the row.

        Args:
            row (int): The row (0-based).
            sink_label (str): The sink label.
            dataframe (pd.DataFrame): The pandas dataframe.
        """
        # Set the text column
        widget = QTableWidgetItem(sink_label)
        widget.setFlags(widget.flags() & ~Qt.ItemIsEditable)
        self.table_widget.setItem(row, self.COL_SINK_LABEL, widget)
        max_delta = '<undefined>'
        if dataframe is not None:
            if len(dataframe) == 0:
                max_delta = '<same>'
            else:
                max_delta = str(max([dataframe['Delta'].max(), dataframe['Delta'].min()], key=abs))
        widget = QTableWidgetItem(max_delta)
        widget.setFlags(widget.flags() & ~Qt.ItemIsEditable)
        self.table_widget.setItem(row, self.COL_MAX_DELTA, widget)

        # Add the View... button
        edit_text = 'View...'
        btn_view = QPushButton(self.table_widget)
        btn_view.setText(edit_text)
        btn_view.clicked.connect(partial(self._on_btn_view, sink_label))
        self.table_widget.setCellWidget(row, self.COL_TIME_SERIES, btn_view)

    def _add_max_delta(self, vertical_layout: QVBoxLayout):
        """Creates the max delta text field.

        Args:
            vertical_layout: The layout
        """
        max_delta = 0.0
        for df in self._dataframes.values():
            if df is not None:
                if len(df) != 0:
                    delta = max([df['Delta'].max(), df['Delta'].min()], key=abs)
                    if abs(delta) > abs(max_delta):
                        max_delta = delta
        self.max_delta_text = QLabel(f'Maximum Delta: {max_delta}')
        vertical_layout.addWidget(self.max_delta_text)

    def _on_btn_view(self, sink_label: str):
        """Called when a View button is clicked.

        Args:
            sink_label (str): The sink label of the row that was clicked.
        """
        app_name = os.environ.get('XMS_PYTHON_APP_NAME')
        df = self._dataframes[sink_label]
        if df is not None:
            if len(df) == 0:
                message_box.message_with_ok(self,
                                            f'The SWMM data and the SRH-2D data at {self._parameter_string} '
                                            f'"{sink_label}" are equal.', app_name)
            else:
                dlg = DataframeViewer(df, f'Current vs New Flow Comparison, {sink_label}', self)
                dlg.exec()
        else:
            message_box.message_with_ok(self, f'The SWMM data and the SRH-2D data at {self._parameter_string} '
                                              f'"{sink_label}" are different sizes and '
                                              'cannot be compared.', app_name)
