"""BridgeFootprint class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from enum import Enum
import logging
import math

# 2. Third party modules
from rtree import index
from shapely.geometry import LineString, MultiPolygon, Point, Polygon
from shapely.ops import nearest_points

# 3. Aquaveo modules
from xms.constraint.ugrid_builder import UGridBuilder
import xms.grid.geometry.geometry as geom
from xms.grid.ugrid import UGrid
import xms.mesher
from xms.mesher import meshing
from xms.tool_core.coverage_builder import CoverageBuilder

# 4. Local modules
from xms.tool.algorithms.coverage.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.tool.algorithms.mesh_2d.piers import (PierGroup, PointedUnitWallPier, rotate_point, RoundUnitWallPier,
                                               SquareUnitWallPier, translate_point)
from xms.tool.algorithms.ugrids.ugrid_2d_merger import UGrid2dMerger, xy_distance


class ArcType(str, Enum):
    """The arc type."""
    BRIDGE = 'Bridge'
    PIER = 'Pier'
    ABUTMENT = 'Abutment'


class PierType(str, Enum):
    """The pier type."""
    WALL = 'Wall'
    GROUP = 'Group'


class PierEndType(str, Enum):
    """The pier end type."""
    SQUARE = 'Square'
    ROUND = 'Round'
    SHARP = 'Sharp'


class BridgePierData:
    """Class for storing the Channel coverage properties."""
    ARC_TYPE_BRIDGE = 1
    ARC_TYPE_PIER = 2
    ARC_TYPE_ABUTMENT = 3
    PIER_TYPE_WALL = 0
    PIER_TYPE_GROUP = 1
    PIER_END_TYPE_SQUARE = 0
    PIER_END_TYPE_ROUND = 1
    PIER_END_TYPE_SHARP = 2


class BridgeFootprint:
    """Class creates ugrid and coverage polygon from bridge and pier information."""

    def __init__(self, arc_data, inputs, logger=None):
        """Initializes the class.

        Args:
            arc_data (list(dict)): arc points and attributes
            inputs (dict): ugrid uuid, mesh transition bias
            logger (logging.logger): An optional logger replacement.
        """
        self.arc_data = arc_data
        self.inputs = inputs
        if logger is None:
            self.logger = logging.getLogger('xms.ewn')
        else:
            self.logger = logger
        self.coverage_elevation = 0.0
        self.out_ugrid = None
        self.out_coverage = None
        self._worker = None
        self.bridge_upstream_line = None
        self.bridge_downstream_line = None
        self._max_iter = 50

    def generate(self):
        """Generates a mesh from the bridge and pier definitions in the coverage."""
        done = False
        width = -1
        wrap_width = -1
        num_iter = 1
        override_widths = []
        while not done:
            try:
                self._worker = _BridgeFootprint(self.arc_data, self.inputs, self.logger, override_widths)
                self._worker.coverage_elevation = self.coverage_elevation
                self._worker.generate()

                done = True
                self.out_ugrid = self._worker.out_ugrid
                self.out_coverage = self._worker.out_coverage
                self.bridge_upstream_line = self._worker.bridge_upstream_line
                self.bridge_downstream_line = self._worker.bridge_downstream_line
            except RuntimeError as err:
                if not self._worker.piers:
                    raise
                if width == -1:
                    orig_width = width = self._worker.bridge_arc['bridge_width']
                    wrap_width = self._worker.bridge_arc['bridge_wrapping_width']
                min_bridge_width = self._worker.piers[0].min_bridge_width() + _BridgeFootprint.TOL
                # if width <= min_bridge_width:
                #     self.logger.warning(err)
                #     raise
                pier_out_wrap_width = self._worker.piers[0].outer_layer_thick[-1]
                new_width = max(float(int(width - (2 * pier_out_wrap_width))), min_bridge_width)
                new_wrap_width = (orig_width - new_width) / 2
                override_widths = [new_width, new_wrap_width, wrap_width]
                width = new_width
                num_iter += 1
                if num_iter > self._max_iter:
                    self.logger.warning(err)
                    self.logger.warning('Exceeded max number of iterations.')
                    raise


class _BridgeFootprint:
    """Class creates ugrid and coverage polygon from bridge and pier information."""
    TOL = 0.1

    def __init__(self, arc_data, inputs, logger=None, widths=None):
        """Initializes the class.

        Args:
            arc_data (list(dict)): arc points and attributes
            inputs (dict): ugrid uuid, mesh transition bias
            logger (logging.logger): An optional logger replacement.
            widths (list): Optional argument to give a list of widths to override bridge width and wrapping width
        """
        self._logger = logger
        self._inputs = inputs
        self._arc_data_list = arc_data
        self._override_widths = widths
        self.coverage_elevation = None
        self.out_ugrid = None
        self.out_coverage = None
        self.bridge_upstream_line = None
        self.bridge_downstream_line = None

        self.bridge_arc = None
        self._bridge_line = None
        self._bridge_offsets = None
        self._bridge_top_bot = []
        self._bridge_offsets_insert_pts = [{}, {}]
        self._bridge_num_segments = None
        self._pier_pts_start_end = []
        self._patch_corner_points = []
        self._corner_pts_rtree = None
        self._meshing_inputs = []
        self._intersecting_piers_polys = []

        self._abutment_arc_0 = None
        self._abutment_arc_1 = None
        self._abutment_uv = [None, None]
        self._pier_arcs = []
        self.piers = []
        self._cur_pier = None
        self._cur_pier_growth_factor = 1.5
        self._first_pier_group_loc_data = None
        self._first_loc_pier_group = None
        self._cur_loc_pier_group = None
        self._cur_pier_in_group_idx = -1

        # by default create the mesh and not the coverage
        self._new_coverage_name = self._inputs.get('new_coverage_name', '')
        self._create_mesh = self._inputs.get('create_mesh', not self._new_coverage_name)
        self._wrap_upstream = self._inputs.get('wrap_upstream', True)
        self._wrap_downstream = self._inputs.get('wrap_downstream', True)
        self._bridge_num_side_elem = self._inputs.get('bridge_num_side_elem', None)
        self._pier_rotate_translate = True
        self._pier_n_layers = 1
        self._polygon_corners = set()
        self._co_grid = None

    def generate(self):
        """Generates a mesh from the bridge and pier definitions in the coverage."""
        self._get_bridge_arc()
        self._get_abutment_arcs()
        self._calc_bridge_offset_arcs()
        self._calc_pier_bridge_intersections()
        self._generate_piers()
        self._calc_bridge_upstream_downstream_patches()
        if self._override_widths:
            widths = self._override_widths
            self.bridge_arc['bridge_wrapping_width'] = widths[2]
            self._override_widths = None
            self._calc_bridge_upstream_downstream_patches()
            self._override_widths = widths
        self._calc_bridge_mesh()
        if self._new_coverage_name:
            self._calc_coverage()
        if not self._create_mesh:
            self.out_ugrid = None
        self.bridge_upstream_line = self._bridge_offsets[0]
        self.bridge_downstream_line = self._bridge_offsets[1]

    def _get_bridge_arc(self):
        """Find the first bridge arc in the coverage. Logs a warning if more than 1 bridge arc exists."""
        for arc_data in self._arc_data_list:
            arc_id = arc_data['id']
            if arc_data['arc_type'] == ArcType.BRIDGE:
                if self.bridge_arc is None:
                    self.bridge_arc = arc_data
                    if self._override_widths:
                        arc_data['bridge_width'] = self._override_widths[0]
                        arc_data['bridge_wrapping_width'] = self._override_widths[1]
                    msg = ''
                    if arc_data['bridge_width'] <= 0.0:
                        msg = f'Bridge width must be greater than 0.0. Arc id: {arc_id} will be ignored.'
                    elif arc_data['bridge_wrapping_width'] <= 0.0:
                        msg = f'Bridge wrapping width must be greater than 0.0. Arc id: {arc_id} will be ignored.'
                    if msg:
                        raise RuntimeError(msg)
                    specify_num_seg = arc_data.get('bridge_specify_num_segments', 0)
                    if specify_num_seg:
                        nseg = arc_data['bridge_num_segments']
                        if nseg < 1:
                            msg = f'Number of upstream/downstream segments for a bridge must be greater than 0. ' \
                                  f'Specified number of segment will be ignored on Arc id: {arc_id}.'
                            raise RuntimeError(msg)
                        else:
                            self._bridge_num_segments = arc_data['bridge_num_segments']
                    # specify number of segments on the bridge center line
                    cl_seg = self._inputs.get('center_line_num_segments', None)
                    if cl_seg is not None:
                        arc_data['arc_pts'] = self._redistribute_line_num_seg(arc_data['arc_pts'], cl_seg)
                else:
                    msg = f'More than one bridge arc found in coverage. Arc id: {arc_id} will be ignored.'
                    self._logger.warning(msg)
        if self.bridge_arc is None:
            msg = 'No bridge arc found in coverage. Aborting.'
            raise RuntimeError(msg)

    def _get_abutment_arcs(self):
        """Get the abutment arcs."""
        self._bridge_line = LineString(self.bridge_arc['arc_pts'])
        for arc_data in self._arc_data_list:
            if arc_data['arc_type'] == ArcType.ABUTMENT:
                abutment_line = LineString((arc_data['arc_pts'][0], arc_data['arc_pts'][-1]))
                crossing = self._bridge_line.intersection(abutment_line)
                arc_id = arc_data['id']
                if not isinstance(crossing, Point):
                    msg = f'Invalid intersection between abutment arc id: {arc_id} and bridge. ' \
                          f'Abutment will be ignored.'
                    self._logger.warning(msg)
                    continue
                if len(arc_data['arc_pts']) > 2:
                    msg = f'Vertices on abutment arc id {arc_id} will be ignored.'
                    self._logger.warning(msg)
                dist0 = crossing.distance(Point(self.bridge_arc['arc_pts'][0]))
                dist1 = crossing.distance(Point(self.bridge_arc['arc_pts'][-1]))
                if dist0 < dist1 and self._abutment_arc_0 is None:
                    self._abutment_arc_0 = arc_data
                elif self._abutment_arc_1 is None:
                    self._abutment_arc_1 = arc_data
                else:
                    msg = f'Two abutment arcs already found. Abutment arc id: {arc_id} will be ignored.'
                    self._logger.warning(msg)

    def _calc_bridge_offset_arcs(self):
        """Calculate the bridge offset arcs."""
        self._calc_abutment_angles()
        bridge_width = self.bridge_arc['bridge_width']
        dist = bridge_width / 2.0
        bridge_pts = [[p[0], p[1], 0.0] for p in self.bridge_arc['arc_pts']]
        tup = self._offset_line(bridge_pts, dist)
        for t in tup:
            ls = LineString(t)
            if not ls.is_simple:
                msg = 'Unable to offset bridge center line. Result is not "simple". ' \
                      'Center line segments have too much curvature.'
                raise RuntimeError(msg)
        self._bridge_offsets = [tup[0], tup[1]]
        self._bridge_offsets_t_vals = (line_to_parametric_values(self._bridge_offsets[0]),
                                       line_to_parametric_values(self._bridge_offsets[1]))

    def _orient_arcs(self, arcs):
        """Make sure that the arcs have the correct orientation relative to the bridge arc.

        Args:
            arcs (list): list of the arcs
        """
        for arc in arcs:
            if arc is None:
                continue
            abut_line = LineString((arc['arc_pts'][0], arc['arc_pts'][-1]))
            for i in range(1, len(self.bridge_arc['arc_pts'])):
                pts = (self.bridge_arc['arc_pts'][i - 1], self.bridge_arc['arc_pts'][i])
                bridge_segment = LineString(pts)
                if bridge_segment.intersects(abut_line):
                    # point 0 of the abutment should be on the left side of the bridge
                    ax, ay, bx, by = pts[0][0], pts[0][1], pts[1][0], pts[1][1]
                    px, py = arc['arc_pts'][0][0], arc['arc_pts'][0][1]
                    cross = ((bx - ax) * (py - ay)) - ((by - ay) * (px - ax))
                    if cross < 0:
                        arc['arc_pts'].reverse()
                    break

    def _calc_abutment_angles(self):
        """Calculate the unit vectors based on the abutment arcs."""
        # if self._abutment_arc_0 is None and self._abutment_arc_1 is None:
        #     return
        # we want the arcs oriented so that the first node is on the left of the bridge and second node
        # is on the right side of the bridge
        arcs = [self._abutment_arc_0, self._abutment_arc_1]
        self._orient_arcs(arcs)

        for i, arc in enumerate(arcs):
            if arc is None:
                if i == 0:
                    prev_pt = None
                    pt = self.bridge_arc['arc_pts'][0]
                    next_pt = self.bridge_arc['arc_pts'][1]
                else:
                    prev_pt = self.bridge_arc['arc_pts'][-2]
                    pt = self.bridge_arc['arc_pts'][-1]
                    next_pt = None
                self._abutment_uv[i] = get_direction_from_edges(prev_pt, pt, next_pt)
            else:
                self._abutment_uv[i] = two_d_unit_vector_from_points(arc['arc_pts'][0], arc['arc_pts'][-1])

    def _offset_line(self, pts, dist):
        """Offset a line by the give distance.

        Args:
            pts (list(x,y,z)): points of the line
            dist (float): distance to offset the line
        Returns:
            tuple(list(x,y,z), list(x,y,z)): right and left offsets of the line
        """
        my_line = LineString(pts)
        left_ls = my_line.parallel_offset(dist, side='left')
        left = []
        left_t = 0.0
        left_checker = {'line_string': left_ls, 'line_pts': left, 'dist_t': left_t}

        right_ls = my_line.parallel_offset(dist)
        # right_pts = [(p[0], p[1]) for p in right_ls.coords]
        # right_pts.reverse()
        # right_ls = LineString(right_pts)
        right = []
        right_t = 0.0
        right_checker = {'line_string': right_ls, 'line_pts': right, 'dist_t': right_t}

        total_length = my_line.length
        t_vals = [0.0] * len(pts)
        lengths = [0.0] * len(pts)
        for i in range(1, len(pts)):
            lengths[i] = lengths[i - 1] + xy_distance(pts[i - 1], pts[i])
            t_vals[i] = lengths[i] / total_length

        prev_pt = None
        uvs = []
        for i, pt in enumerate(pts):
            if i + 1 < len(pts):
                next_pt = pts[i + 1]
            else:
                next_pt = None
            uvs.append(get_direction_from_edges(prev_pt, pt, next_pt))
            prev_pt = pt

        def _check_last_point(checker):
            pt = checker['line_pts'][-1]
            sh_pt = Point((pt[0], pt[1]))
            dist2 = checker['line_string'].project(sh_pt, normalized=True)
            if dist2 < checker['dist_t']:
                checker['dist_t'] += 0.01
                sh_pt = checker['line_string'].interpolate(checker['dist_t'], normalized=True)
                checker['line_pts'][-1] = [sh_pt.coords[0][0], sh_pt.coords[0][1], pt[2]]
            else:
                checker['dist_t'] = dist2

        for pt, uv, t_val in zip(pts, uvs, t_vals):
            uv2 = self._adjust_uv(uv, t_val)
            dot = uv[0] * uv2[0] + uv[1] * uv2[1]
            dist1 = dist / dot
            right.append([pt[0] + uv2[0] * dist1, pt[1] + uv2[1] * dist1, pt[2]])
            _check_last_point(right_checker)

            left.append([pt[0] + uv2[0] * dist1 * -1, pt[1] + uv2[1] * dist1 * -1, pt[2]])
            _check_last_point(left_checker)

        return right, left

    def _adjust_uv(self, uv, t_val):
        """Adjust the offset vector based on the abutment arc vectors.

        Arguments:
            uv (x, y): the unit vector in the x,y direction
            t_val (float): parametric value along the line (0.0-1.0)

        Returns:
            tuple (x, y): the updated unit vector
        """
        if self._abutment_arc_0 is None and self._abutment_arc_1 is None:
            return uv
        dx = self._abutment_uv[1][0] - self._abutment_uv[0][0]
        dy = self._abutment_uv[1][1] - self._abutment_uv[0][1]
        abut_uv = (self._abutment_uv[0][0] + (t_val * dx),
                   self._abutment_uv[0][1] + (t_val * dy))
        return abut_uv

    def _calc_pier_bridge_intersections(self):
        """Find the intersections between the pier arcs and the bridge.

        Log warnings if the pier does not intersect the bridge.
        """
        for arc_data in self._arc_data_list:
            if arc_data['arc_type'] == ArcType.PIER:
                pier_line = LineString(arc_data['arc_pts'])
                crossing = self._bridge_line.intersection(pier_line)
                arc_id = arc_data['id']
                if not isinstance(crossing, Point):
                    msg = f'Invalid intersection between pier arc id: {arc_id} and bridge. Pier will be ignored.'
                    self._logger.warning(msg)
                    continue
                if len(arc_data['arc_pts']) > 2:
                    msg = f'Vertices on pier arc id {arc_id} will be ignored.'
                    self._logger.warning(msg)
                self._pier_arcs.append({'arc_data': arc_data,
                                        'crossing_coords': crossing.coords[0]})

    def _generate_piers(self):
        """Create mesh elements from the piers."""
        remove_piers = []
        pier_polys = []
        cnt_intersecting_pier_patches = 0
        abut_lines = []
        if self._bridge_offsets is not None:
            abut_lines = [LineString([self._bridge_offsets[0][0], self._bridge_offsets[1][0]]),
                          LineString([self._bridge_offsets[0][-1], self._bridge_offsets[1][-1]])]
        for pier in self._pier_arcs:
            self._cur_pier_growth_factor = 1.5
            self._cur_pier = pier
            arc_data = self._cur_pier['arc_data']
            arc_id = arc_data['id']
            if 'angle' not in arc_data:
                arc_data['angle'] = calc_arc_angle([arc_data['arc_pts'][0], arc_data['arc_pts'][-1]])
            my_pier = self._calc_pier_and_growth_factor()

            new_pier_patch = len(self._intersecting_piers_polys) > cnt_intersecting_pier_patches

            if my_pier is not None or new_pier_patch:
                if new_pier_patch:
                    poly = self._intersecting_piers_polys[-1]
                    cnt_intersecting_pier_patches += 1
                else:
                    self.piers.append(my_pier)
                    poly = my_pier.outer_polygon()
                    poly.append(poly[0])
                sh_poly = Polygon(poly)
                for item in pier_polys:
                    arc_id1 = item[0]
                    pp = item[1]
                    if sh_poly.intersects(pp) or sh_poly.distance(pp) < self.TOL:
                        msg = f'Piers associated with arc ids: {arc_id}, {arc_id1} intersect. Aborting.'
                        raise RuntimeError(msg)
                for item in abut_lines:
                    if sh_poly.intersects(item):
                        msg = f'Pier associated with arc id: {arc_id} intersects the bridge abutment. Aborting.'
                        raise RuntimeError(msg)
                pier_polys.append((arc_id, sh_poly))
            else:
                remove_piers.append(pier)

        for pier in remove_piers:
            self._pier_arcs.remove(pier)

        if self._pier_arcs:
            self._redistribute_bridge_offsets()

    def _redistribute_bridge_offsets(self):
        """Redistribute the points on the bridge offset lines based on points from the piers."""
        arc_min_max_t = self._calc_peir_arc_min_max_t()
        self._insert_pier_points_into_bridge_offsets(arc_min_max_t)
        self._redist_non_pier_pts()
        self._calc_bridge_top_bot(arc_min_max_t)
        self._calc_patches_between_piers()

    def _calc_bridge_footprint_polygon(self):
        """Calculates the bridge footprint polygon.

        Returns:
            (Polygon): shapely Polygon
        """
        right = self._bridge_offsets[0]
        left = self._bridge_offsets[1].copy()
        left.reverse()
        bot = self._bridge_top_bot[1][1:-1]
        top = self._bridge_top_bot[0][1:-1]
        top.reverse()
        return Polygon(right + bot + left + top)

    def _calc_pier_polygons(self):
        """Creates a shapely multipolygon from the piers.

        Returns:
            (Multipolygon): a multipolygon of the piers.
        """
        polys = []
        for pier in self.piers:
            poly = pier.outer_polygon()
            polys.append(Polygon(poly + [poly[0]]))
        for poly in self._intersecting_piers_polys:
            polys.append(Polygon(poly))
        return MultiPolygon(polys)

    def _is_corner_point(self, pt):
        """Determines if a point is a corner point in a patch.

        Args:
            pt: location x, y, z

        Returns:
            (bool): True if this is a patch corner point
        """
        if self._corner_pts_rtree is None:
            self._corner_pts_rtree = index.Index()
            for i, p in enumerate(self._patch_corner_points):
                self._corner_pts_rtree.insert(i, (p[0], p[1], p[0], p[1]))
        idx = list(self._corner_pts_rtree.nearest((pt[0], pt[1]), 1))[0]
        pc = self._patch_corner_points[idx]
        dist = xy_distance(pt, pc)
        if dist < 1.0e-9:
            return True
        return False

    def _calc_patches_between_piers(self):
        """Create the patch meshing inputs between piers."""
        bridge_poly = self._calc_bridge_footprint_polygon()
        pier_polys = self._calc_pier_polygons()
        multi_poly = bridge_poly.difference(pier_polys)
        for poly in multi_poly.geoms:
            patch = [[p[0], p[1], 0.0] for p in list(poly.exterior.coords)]
            patch.pop()
            if not self._is_corner_point(patch[0]):
                for i, p in enumerate(patch):
                    if self._is_corner_point(p):
                        patch = patch[i:] + patch[:i]
                        break
                if not self._is_corner_point(patch[0]):  # pragma no cover - not sure how to hit this
                    raise RuntimeError('First point is not a patch corner. Aborting.')
            # find the other corners
            corners = []
            for i in range(1, len(patch)):
                p = patch[i]
                if self._is_corner_point(p):
                    corners.append(i)
                    if len(corners) == 3:
                        break
            if len(corners) < 3:  # pragma no cover - not sure how to hit this
                raise RuntimeError('Unable to find 4 corners for patch.')
            m = meshing.PolyInput(outside_polygon=patch, polygon_corners=corners)
            ug = xms.mesher.generate_mesh([m])
            if ug.cell_count > 0:
                self._meshing_inputs.append(meshing.PolyInput(outside_polygon=patch,
                                                              polygon_corners=corners))
            # we don't have a test case that causes this issue
            # else:
            #     self._meshing_inputs.append(meshing.PolyInput(outside_polygon=patch))

    def _calc_bridge_top_bot(self, arc_min_max_t):
        """Create lines for the bridge top and bottom.

        Args:
            arc_min_max_t (list(dict)): list of dicts key=arc_id, value=list(t_values)
        """
        min_t = 1.0
        max_t = 0.0
        num_side_elements = [1, 1]
        for pier_arc in self._pier_arcs:
            arc_id = pier_arc['arc_data']['id']
            if self._bridge_num_side_elem:
                n_side_elements = self._bridge_num_side_elem
            elif pier_arc['arc_data']['pier_type'] == PierType.WALL:
                n_side_elements = max(1, pier_arc['arc_data']['pier_num_side_elements'])
            else:
                if pier_arc['arc_data']['number_piers'] == 1:
                    n_side_elements = 2
                else:
                    n_elem_between_piers = pier_arc['arc_data'].get('num_elements_between_piers', 1)
                    n_side_elements = (pier_arc['arc_data']['number_piers'] - 1) * n_elem_between_piers
            if arc_min_max_t[0][arc_id][0] < min_t:
                min_t = arc_min_max_t[0][arc_id][0]
                num_side_elements[0] = n_side_elements
            if arc_min_max_t[0][arc_id][-1] > max_t:
                max_t = arc_min_max_t[0][arc_id][-1]
                num_side_elements[1] = n_side_elements
        lines = [[self._bridge_offsets[0][0], self._bridge_offsets[1][0]],
                 [self._bridge_offsets[0][-1], self._bridge_offsets[1][-1]]]
        for i, line in enumerate(lines):
            nseg = num_side_elements[i]
            dx = line[1][0] - line[0][0]
            dy = line[1][1] - line[0][1]
            factor = 1.0 / nseg
            new_line = [line[0]]
            for j in range(1, nseg):
                x = line[0][0] + (j * factor * dx)
                y = line[0][1] + (j * factor * dy)
                new_line.append([x, y, line[0][2]])
            new_line.append(line[-1])
            lines[i] = new_line
        self._bridge_top_bot = lines

    def _calc_peir_arc_min_max_t(self):
        """Calculates the min/max t value for each pier arc relative to the bridge offset lines.

        Returns:
            (list(dict)): list of dicts key=arc_id, value=list(t_values)
        """
        arc_min_max_t = [{}, {}]
        for i in [0, 1]:
            my_d = self._bridge_offsets_insert_pts[i]
            keys = sorted(my_d.keys())
            self._bridge_offsets_insert_pts[i] = [(k, my_d[k][0], my_d[k][1]) for k in keys]
            arc_id_to_t_vals = {}
            # for each arc_id get the min t_val and the max t_val
            for item in self._bridge_offsets_insert_pts[i]:
                t_val = item[0]
                arc_id = item[2]
                arc_id_to_t_vals.setdefault(arc_id, [])
                arc_id_to_t_vals[arc_id].append(t_val)
            for k, v in arc_id_to_t_vals.items():
                v.sort()
                arc_min_max_t[i][k] = (v[0], v[-1])
        return arc_min_max_t

    def _insert_pier_points_into_bridge_offsets(self, arc_min_max_t):
        """Insert the pier points that were moved to the bridge offset lines.

        Args:
            arc_min_max_t (list(dict)): list of dicts key=arc_id, value=list(t_values)
        """
        for i in [0, 1]:
            new_line = [self._bridge_offsets[i][0]]
            self._patch_corner_points.append(new_line[0])
            insert_list = self._bridge_offsets_insert_pts[i]
            insert_idx = 0
            insert_t = insert_list[0][0]
            pts_start_end = []
            j = 1
            while j < len(self._bridge_offsets[i]):
                t0 = self._bridge_offsets_t_vals[i][j - 1]
                t1 = self._bridge_offsets_t_vals[i][j]
                if t0 < insert_t < t1:
                    arc_id = insert_list[insert_idx][2]
                    max_t = arc_min_max_t[i][arc_id][1]
                    # skip points on the line that are between the pier points
                    while self._bridge_offsets_t_vals[i][j] < max_t:
                        j += 1
                    # insert the pier points
                    pts_start_end.append([len(new_line), -1])
                    while insert_idx < len(insert_list) and insert_list[insert_idx][2] == arc_id:
                        new_line.append(insert_list[insert_idx][1])
                        insert_idx += 1
                    pts_start_end[-1][1] = len(new_line) - 1
                    self._patch_corner_points.append(new_line[pts_start_end[-1][0]])
                    self._patch_corner_points.append(new_line[-1])
                    if insert_idx < len(insert_list):
                        insert_t = insert_list[insert_idx][0]
                    else:
                        insert_t = 1.1  # no more pts to insert
                else:
                    new_line.append(self._bridge_offsets[i][j])
                    j += 1
            self._pier_pts_start_end.append(pts_start_end)
            self._bridge_offsets[i] = new_line
            self._patch_corner_points.append(new_line[-1])

    def _calc_num_segments(self, start_ends):
        """Calculate the number of segments on the mesh between each pier.

        Args:
            start_ends (tuple(list)): the start and end indexes on the up and down offset from
                                      the bridge center line.

        Returns:
            (list): number of segments for each section
        """
        nseg = []
        for idx, _ in enumerate(start_ends[0]):
            up_start_end = start_ends[0][idx]
            dn_start_end = start_ends[1][idx]
            # make a polygon from these points
            pts = [self._bridge_offsets[0][up_start_end[0]],
                   self._bridge_offsets[1][dn_start_end[0]],
                   self._bridge_offsets[1][dn_start_end[1]],
                   self._bridge_offsets[0][up_start_end[1]],
                   ]
            pts = [(p[0], p[1]) for p in pts]
            poly = Polygon(pts)
            # intersect the bridge center line
            new_line = self._bridge_line.intersection(poly)
            # count the number of segments
            nseg.append(len(new_line.coords) - 1)
        return nseg

    def _redist_non_pier_pts(self):
        """Redistribute the points between the piers on the bridge offset lines."""
        start_ends = [[], []]  # compute indexes of the gaps
        for i in [0, 1]:
            start = 0
            for start_end in self._pier_pts_start_end[i]:
                end = start_end[0]
                start_ends[i].append([start, end])
                start = start_end[1]
            start_ends[i].append([start, len(self._bridge_offsets[i]) - 1])

        nsegs = self._calc_num_segments(start_ends)
        # nsegs = []
        # for i in [0, 1]:
        #     for idx, start_end in enumerate(start_ends[i]):
        #         start = start_end[0]
        #         end = start_end[1]
        #         nseg = end - start
        #         if i == 0:
        #             nsegs.append(nseg)
        #         else:
        #             nsegs[idx] = int((nseg + nsegs[idx]) / 2)

        for i in [0, 1]:
            new_offset = []
            for idx, start_end in enumerate(start_ends[i]):
                start = start_end[0]
                end = start_end[1]
                nseg = nsegs[idx]
                if idx > 0:
                    last_end = start_ends[i][idx - 1][1]
                    new_offset.extend(self._bridge_offsets[i][last_end + 1:start])
                p0 = self._bridge_offsets[i][start]
                p1 = self._bridge_offsets[i][end]
                dx = p1[0] - p0[0]
                dy = p1[1] - p0[1]
                new_offset.append(p0)
                t_increment = 1.0 / nseg
                for j in range(1, nseg):
                    p = [0.0, 0.0, 0.0]
                    p[0] = p0[0] + ((j * t_increment) * dx)
                    p[1] = p0[1] + ((j * t_increment) * dy)
                    new_offset.append(p)
                new_offset.append(p1)
            self._bridge_offsets[i] = new_offset

    def _calc_pier_and_growth_factor(self):
        """Calculates a pier and determines the growth factor such that the pier will extend to the bridge footprint.

        Returns:
            (PierBase): the returned class
        """
        arc_data = self._cur_pier['arc_data']
        arc_id = arc_data['id']
        self._calc_pier_num_wrap_layers()
        pier_n_layers = self._pier_n_layers
        self._pier_n_layers = 1
        pier = self._calc_pier()
        self._pier_n_layers = pier_n_layers
        if pier is None:
            return None
        pier_poly = pier.outer_polygon()
        # calculate the distance from the pier to the bridge footprint
        left_offset = LineString(self._bridge_offsets[1])
        pp = Polygon(pier_poly + [pier_poly[0]])
        if left_offset.intersects(pp):
            if arc_data['pier_type'] == PierType.GROUP:
                msg = f'Pier intersects bridge footprint. Arc id: {arc_id} will be ignored.'
                self._logger.warning(msg)
                return None
            else:
                self._create_pier_intersects_bridge()
                return None
        pier = self._iterative_calc_pier()
        pier = self._move_pier_nodes_to_bridge_footprint(pier)
        return pier

    def _calc_pier_num_wrap_layers(self):
        """Calculates the number of layers for a pier based on the wrapping width and a growth factor of 1.5."""
        self._pier_rotate_translate = False
        dist = 0.5 * self.bridge_arc['bridge_width']
        self._pier_n_layers = 0
        dist_diff = None
        done = False
        while not done:
            self._pier_n_layers += 1
            pier = self._calc_pier()
            if pier is None:
                return
            pier_poly = pier.outer_polygon()
            y_vals = [p[1] for p in pier_poly]
            max_y_val = max(y_vals)
            diff = abs(max_y_val - dist)
            if dist_diff is None or diff < dist_diff:
                dist_diff = diff
            else:
                self._pier_n_layers -= 1
                done = True
        p_type = self._cur_pier['arc_data']['pier_type']
        if p_type == PierType.GROUP and self._pier_n_layers < 2:
            self._pier_n_layers = 2
        self._pier_rotate_translate = True

    def _calc_intersecting_pier_offsets(self):
        """Calculates widths from the pier center line for wrapping elements.

        Returns:
            (list): list of widths
        """
        arc_data = self._cur_pier['arc_data']
        # when a wall pier goes past the bridge edges you only get one
        # for loop below only loops once
        # also see comment in _add_intersecting_pier_to_patches
        w = arc_data['pier_size']
        pier_wrap_width = arc_data.get('pier_element_wrap_width', w)
        half_w = 0.5 * w
        widths = [half_w, half_w + pier_wrap_width]
        # widths = [0.5 * w]
        # new_width = widths[-1]
        # nlay = 1
        # for i in range(nlay):
        #     new_width = new_width + 1.5 ** (i + 1) * w
        #     widths.append(new_width)
        return widths

    def _calc_intersecting_pier_line(self, loc):
        """Calculates the pier line that extends past the bridge footprint.

        Args:
            loc (x, y) : location of the pier

        Returns:
            (list): list of the 2 end points of the pier line that extends beyond the bridge footprint
        """
        arc_data = self._cur_pier['arc_data']
        line_dist = self.bridge_arc['bridge_width'] * 5
        angle = math.radians(arc_data['angle'])
        cos_rad = math.cos(angle)
        sin_rad = math.sin(angle)
        if loc is None:
            loc = self._cur_pier['crossing_coords']
        p0 = [0.0, line_dist, 0.0]
        p1 = [0.0, -line_dist, 0.0]
        rotate_point(p0, cos_rad, sin_rad)
        translate_point(p0, loc)
        rotate_point(p1, cos_rad, sin_rad)
        translate_point(p1, loc)
        d0 = xy_distance(p0, arc_data['arc_pts'][0])
        d1 = xy_distance(p0, arc_data['arc_pts'][-1])
        if d0 < d1:
            tmp = p1
            p1 = p0
            p0 = tmp
        return [p0, p1]

    def _offset_intersecting_pier_line(self, pier_line, widths):
        """Creates offsets for the pier_line by the distances defined in widths.

        Args:
            pier_line (list): the line defining the pier
            widths (list): offset distances from the pier

        Returns:
            (tuple(lists): tuple of lists of the offset lines trimmed to the bridge footprint
        """
        arc_data = self._cur_pier['arc_data']
        arc_id = arc_data['id']
        offset_right = []
        offset_left = []
        bright = LineString(self._bridge_offsets[0])
        bleft = LineString(self._bridge_offsets[1])
        sh_pier = LineString(pier_line)
        for width in widths:
            lines = sh_pier.parallel_offset(width, 'right'), sh_pier.parallel_offset(width, 'left')
            # trim lines to bridge offsets
            ip0, _ = nearest_points(bright, lines[0])
            ip1, _ = nearest_points(bleft, lines[0])
            ip0 = [ip0.coords[0][0], ip0.coords[0][1], 0.0]
            ip1 = [ip1.coords[0][0], ip1.coords[0][1], 0.0]
            line0 = [ip0, ip1]
            t = point_t_value_on_line(ip0, self._bridge_offsets[0], self._bridge_offsets_t_vals[0])
            self._bridge_offsets_insert_pts[0][t] = (ip0, arc_id)
            t = point_t_value_on_line(ip1, self._bridge_offsets[1], self._bridge_offsets_t_vals[1])
            self._bridge_offsets_insert_pts[1][t] = (ip1, arc_id)

            ip0, _ = nearest_points(bright, lines[1])
            ip1, _ = nearest_points(bleft, lines[1])
            ip0 = [ip0.coords[0][0], ip0.coords[0][1], 0.0]
            ip1 = [ip1.coords[0][0], ip1.coords[0][1], 0.0]
            line1 = [ip0, ip1]
            t = point_t_value_on_line(ip0, self._bridge_offsets[0], self._bridge_offsets_t_vals[0])
            self._bridge_offsets_insert_pts[0][t] = (ip0, arc_id)
            t = point_t_value_on_line(ip1, self._bridge_offsets[1], self._bridge_offsets_t_vals[1])
            self._bridge_offsets_insert_pts[1][t] = (ip1, arc_id)

            offset_right.append(line0)
            offset_left.append(line1)
        # redistribute points on the first and last lines
        arc_data = self._cur_pier['arc_data']
        n_side_elem = arc_data['pier_num_side_elements']
        for i in [0, -1]:
            offset_right[i] = subdivide_line_segment(offset_right[i][0], offset_right[i][1], n_side_elem)
            offset_left[i] = subdivide_line_segment(offset_left[i][0], offset_left[i][1], n_side_elem)
        return offset_right, offset_left

    def _add_intersecting_pier_to_patches(self, offsets):
        """Adds the intersecting pier polygons to the meshing inputs as patches.

        Args:
            offsets (list): Offset lines from the wall pier
        """
        patches = []
        corners_list = []
        for cnt, lines in enumerate(offsets):
            if cnt == 1:
                lines.reverse()
            patch = []
            for i in range(1, len(lines)):
                patch.append(lines[-i][0])
            corners = [len(patch)]
            patch.extend(lines[0])
            corners.append(len(patch) - 1)
            # when a wall pier goes past the bridge edges you only get one
            # see nlay = 1 line in _calc_intersecting_pier_offsets
            # for i in range(1, len(lines) - 1):
            #     patch.append(lines[i][-1])
            corners.append(len(patch))
            last = lines[-1].copy()
            last.reverse()
            last.pop()
            patch.extend(last)
            patches.append(patch)
            corners_list.append(corners)
            self._meshing_inputs.append(meshing.PolyInput(outside_polygon=patch,
                                                          polygon_corners=corners))
        # create polygon for outside of pier
        idx = corners_list[0][0]  # first corner on first patch
        poly = patches[0][:idx + 1]
        idx = corners_list[1][2]  # last corner on second patch
        poly.extend(patches[1][:idx + 1])
        idx = corners_list[0][1]  # second corner on first patch
        poly.extend(patches[0][idx:])
        self._intersecting_piers_polys.append(poly)

    def _create_pier_intersects_bridge(self):
        """Create a square wall pier that intersects the bridge footprint.

        Returns:
            (PierBase): the pier
        """
        arcs = [self._cur_pier['arc_data']]
        self._orient_arcs(arcs)
        widths = self._calc_intersecting_pier_offsets()
        pier_line = self._calc_intersecting_pier_line(None)
        offsets = self._offset_intersecting_pier_line(pier_line, widths)
        self._add_intersecting_pier_to_patches(offsets)

    def _iterative_calc_pier(self):
        """Iteratively calculate the pier so that it is within 0.01 distance of the bridge footprint.

        Returns:
            (PierBase): the pier
        """
        # self._calc_pier_num_wrap_layers()
        left_offset = LineString(self._bridge_offsets[1])
        done = False
        max_factor = None
        min_factor = None
        while not done:
            pier = self._calc_pier()
            pier_poly = pier.outer_polygon()
            if self._pier_n_layers == 1:
                done = True
                continue
            pp = Polygon(pier_poly + [pier_poly[0]])
            if left_offset.intersects(pp):
                if max_factor is None or self._cur_pier_growth_factor < max_factor:
                    max_factor = self._cur_pier_growth_factor
                if min_factor is None:
                    self._cur_pier_growth_factor *= 0.9
                else:
                    self._cur_pier_growth_factor = 0.5 * (min_factor + self._cur_pier_growth_factor)
            else:
                dist = left_offset.distance(pp)
                if dist < 0.01:
                    done = True
                else:
                    if min_factor is None or self._cur_pier_growth_factor > min_factor:
                        min_factor = self._cur_pier_growth_factor
                    if max_factor is None:
                        self._cur_pier_growth_factor *= 1.1
                    else:
                        self._cur_pier_growth_factor = 0.5 * (max_factor + self._cur_pier_growth_factor)
        if self._cur_pier_growth_factor < 1.0:
            arc_data = self._cur_pier['arc_data']
            arc_id = arc_data['id']
            msg = f'The growth rate for elements wrapping around the pier associated with arc id: ' \
                  f'{arc_id} is less than 1.0. Consider using a smaller wrapping width on this pier.'
            self._logger.info(msg)

        return pier

    def _move_pier_nodes_to_bridge_footprint(self, pier):
        """Move the pier nodes to be on the lines of the bridge footprint.

        Arguments:
            pier (PierBase): the pier

        Returns:
            (PierBase): the updated pier
        """
        arc_data = self._cur_pier['arc_data']
        arc_id = arc_data['id']
        pp = pier.outer_polygon()
        end_pt_idx = pier.end_pt_indexes()
        lines = [LineString(self._bridge_offsets[0]), LineString(self._bridge_offsets[1])]
        for cnt in [0, 1]:
            line = lines[cnt]
            offset = self._bridge_offsets[cnt]
            t_vals = self._bridge_offsets_t_vals[cnt]
            dist = []
            for i in range(len(pp)):
                p = Point(pp[i][0], pp[i][1])
                dist.append((line.distance(p), i))
            dist.sort()
            # get the closest point to the line and now get the next closest points to that point
            idx = dist[0][1]
            if idx in end_pt_idx[0]:
                pt_idxs = end_pt_idx[0]
            elif idx in end_pt_idx[1]:
                pt_idxs = end_pt_idx[1]
            else:  # we don't have a test case for this so we are not worried about coverage
                raise RuntimeError('Unable to find pier point index.')  # pragma no cover
            pts = [pp[i] for i in pt_idxs]
            # move these points along the same vector as that pier
            for p in pts:
                proj_line = self._calc_intersecting_pier_line(p)
                ls = LineString(proj_line)
                sh_pt, _ = nearest_points(line, ls)
                sh_pt = sh_pt.coords[0]
                pt_idx = pier.pts.index(p)
                pier.pts[pt_idx] = [sh_pt[0], sh_pt[1], pp[idx][2]]
                t_val = point_t_value_on_line(pier.pts[pt_idx], offset, t_vals)
                self._bridge_offsets_insert_pts[cnt][t_val] = (pier.pts[pt_idx], arc_id)
        return pier

    def _calc_pier(self):
        """Generate a pier from the current pier.

        Returns:
            (PierBase): the returned class
        """
        loc = self._cur_pier['crossing_coords']
        arc_data = self._cur_pier['arc_data']
        arc_id = arc_data['id']
        angle = arc_data['angle']
        pier_type = arc_data['pier_type']
        pier_size = arc_data['pier_size']
        pier_n_layers = self._pier_n_layers
        pier_wrap_width = arc_data.get('pier_element_wrap_width', 0.0)
        pier_num_side_elements = -1
        if self._bridge_num_side_elem:
            pier_num_side_elements = self._bridge_num_side_elem
        if pier_type == PierType.WALL:
            pier_length = arc_data['pier_length']
            pier_end_type = arc_data['pier_end_type']
            if pier_num_side_elements == -1:
                pier_num_side_elements = arc_data['pier_num_side_elements']
            if pier_end_type == PierEndType.SQUARE:
                my_pier = SquareUnitWallPier(length=pier_length, width=pier_size,
                                             wrap_width=pier_wrap_width,
                                             number_of_side_elements=pier_num_side_elements,
                                             number_layers=pier_n_layers,
                                             growth_factor=self._cur_pier_growth_factor)
            elif pier_end_type == PierEndType.ROUND:
                my_pier = RoundUnitWallPier(length=pier_length, width=pier_size,
                                            wrap_width=pier_wrap_width,
                                            number_of_side_elements=pier_num_side_elements,
                                            number_layers=pier_n_layers,
                                            growth_factor=self._cur_pier_growth_factor)
            else:
                my_pier = PointedUnitWallPier(length=pier_length, width=pier_size,
                                              wrap_width=pier_wrap_width,
                                              number_of_side_elements=pier_num_side_elements,
                                              number_layers=pier_n_layers,
                                              growth_factor=self._cur_pier_growth_factor)
        else:
            number_piers = arc_data['number_piers']
            pier_spacing = arc_data['pier_spacing']
            if pier_spacing < pier_size + 2 * pier_wrap_width:
                msg = f'Pier spacing must be larger than the pier size (diameter) + 2 * wrapping width. ' \
                      f'Skipped arc id: {arc_id}.'
                self._logger.warning(msg)
                return None
            my_pier = PierGroup(number_piers=number_piers, pier_spacing=pier_spacing, diameter=pier_size,
                                wrap_width=pier_wrap_width, number_of_layers=pier_n_layers + 1,
                                growth_factor=self._cur_pier_growth_factor,
                                num_side_elements=pier_num_side_elements)
        if self._pier_rotate_translate:
            my_pier.rotate_points(angle)
            my_pier.translate_points(loc)

        return my_pier

    def _calc_bridge_mesh(self):
        """Creates the bridge mesh if we are inserting the bridge."""
        buffer = self._inputs.get('buffer', self.TOL)
        b = UGridBuilder()
        b.set_is_2d()
        bridge_ug = xms.mesher.generate_mesh(polygon_inputs=self._meshing_inputs)
        b.set_ugrid(bridge_ug)
        bridge_grid = b.build_grid()
        co_grid = bridge_grid

        for pier in self.piers:
            ug = UGrid(pier.pts, pier.cells)
            b.set_ugrid(ug)
            co_pier = b.build_grid()
            merger = UGrid2dMerger(co_pier, co_grid, poly_buffer=buffer)
            merger.merge_grids()
            b.set_ugrid(merger.out_ugrid)
            co_grid = b.build_grid()

        if co_grid is not None:
            self._co_grid = co_grid
            self.out_ugrid = co_grid.ugrid

    def _calc_bridge_patch_no_piers(self):
        """Calculate the bridge footprint patch with no piers defined."""
        right = self._bridge_offsets[0]
        left = self._bridge_offsets[1].copy()
        left.reverse()
        rlen = LineString(right).length / (len(right) - 1)
        llen = LineString(left).length / (len(left) - 1)
        ave_len = 0.5 * (rlen + llen)
        bot_dist = xy_distance(left[0], right[-1])
        top_dist = xy_distance(left[-1], right[0])
        min_dist = min(bot_dist, top_dist)
        nseg = int(min_dist / ave_len) + 1

        dx = left[0][0] - right[-1][0]
        dy = left[0][1] - right[-1][1]
        factor = 1.0 / nseg
        bot = []
        for i in range(1, nseg):
            x = right[-1][0] + (i * factor * dx)
            y = right[-1][1] + (i * factor * dy)
            bot.append([x, y, 0.0])

        dx = right[0][0] - left[-1][0]
        dy = right[0][1] - left[-1][1]
        factor = 1.0 / nseg
        top = []
        for i in range(1, nseg):
            x = left[-1][0] + (i * factor * dx)
            y = left[-1][1] + (i * factor * dy)
            top.append([x, y, 0.0])
        patch = right + bot + left + top + [right[0]]
        patch.reverse()
        patch.pop()
        c0 = len(top) + 1
        c1 = c0 + len(left) - 1
        c2 = c1 + len(bot) + 1
        corners = [c0, c1, c2]
        self._meshing_inputs.append(meshing.PolyInput(outside_polygon=patch,
                                                      polygon_corners=corners))

    def _redistribute_bridge_upstream_downstream(self, pts):
        """Redistribute the upstream or downstream line for the bridge based on user input.

        Args:
            pts (list): list of points that make up the line

        Returns:
            (list): new list of points
        """
        if self._bridge_num_segments is None or self._override_widths:
            return pts
        return self._redistribute_line_num_seg(pts, self._bridge_num_segments)
        # ls = LineString(pts)
        # interp_len = ls.length / self._bridge_num_segments
        # new_pts = [pts[0]]
        # for i in range(self._bridge_num_segments - 1):
        #     the_len = (i + 1) * interp_len
        #     loc = ls.interpolate(the_len)
        #     new_pts.append([loc.x, loc.y, 0.0])
        # new_pts.append(pts[-1])
        # return new_pts

    def _redistribute_line_num_seg(self, pts, num_seg):
        """Create a line with a number of segments given a line.

        Args:
            pts (list): list of points that make up the line
            num_seg (int): number of segments

        Returns:
            (list): new list of points
        """
        ls = LineString(pts)
        interp_len = ls.length / num_seg
        new_pts = [pts[0]]
        for i in range(num_seg - 1):
            the_len = (i + 1) * interp_len
            loc = ls.interpolate(the_len)
            new_pts.append([loc.x, loc.y, 0.0])
        new_pts.append(pts[-1])
        return new_pts

    def _calc_bridge_upstream_downstream_patches(self):
        """Create patches from bridge upstream/downstream polygons."""
        if len(self._meshing_inputs) < 1:  # create a patch for the bridge footprint
            self._calc_bridge_patch_no_piers()

        # make the entry and exit patches
        dist = self.bridge_arc['bridge_wrapping_width']
        offset = self._offset_line(self._bridge_offsets[0], dist)
        new_bridge_offset = [offset[0].copy()]
        offset = (self._redistribute_bridge_upstream_downstream(offset[0]), offset[1])
        offset[0].reverse()
        patch = self._bridge_offsets[0] + offset[0]
        n_pts = len(self._bridge_offsets[0])
        corners = [n_pts - 1, n_pts, len(patch) - 1]
        if not self._override_widths:
            corner_idx = n_pts
            self._polygon_corners.add((patch[corner_idx][0], patch[corner_idx][1]))
            corner_idx = len(patch) - 1
            self._polygon_corners.add((patch[corner_idx][0], patch[corner_idx][1]))
        self._meshing_inputs.append(meshing.PolyInput(outside_polygon=patch,
                                                      polygon_corners=corners))
        if not self._override_widths and not self._wrap_upstream:
            self._meshing_inputs.pop()
        offset = self._offset_line(self._bridge_offsets[1], dist)
        offset = (offset[0], self._redistribute_bridge_upstream_downstream(offset[1]))
        new_bridge_offset.append(offset[1].copy())
        offset[1].reverse()
        patch = self._bridge_offsets[1] + offset[1]
        n_pts = len(offset[1])
        corners = [n_pts - 1, n_pts, len(patch) - 1]
        if not self._override_widths:
            corner_idx = len(offset[0])
            self._polygon_corners.add((patch[corner_idx][0], patch[corner_idx][1]))
            corner_idx = len(patch) - 1
            self._polygon_corners.add((patch[corner_idx][0], patch[corner_idx][1]))
        self._meshing_inputs.append(meshing.PolyInput(outside_polygon=patch,
                                                      polygon_corners=corners))
        if not self._override_widths and not self._wrap_downstream:
            self._meshing_inputs.pop()
        if self._override_widths:
            self._bridge_offsets = new_bridge_offset

    def _calc_coverage(self):
        """Create a coverage from the bridge polygons with holes for the piers."""
        self._calc_bridge_cov_arcs()
        self.out_coverage = self._coverage_builder.build_coverage()

    def _calc_bridge_cov_arcs(self):
        """Creates arcs from the bridge polygons."""
        co_grid = self._co_grid
        ug = co_grid.ugrid
        cell_classes = [0] * ug.cell_count
        poly_builder = GridCellToPolygonCoverageBuilder(co_grid=co_grid,
                                                        dataset_values=cell_classes,
                                                        wkt=None, coverage_name='temp')
        out_polys = poly_builder.find_polygons()
        ug_locs = ug.locations
        poly = [[ug_locs[i][0], ug_locs[i][1], ug_locs[i][2]] for i in out_polys[0][0][0]]
        poly.pop()
        poly = [[p[0], p[1], self.coverage_elevation] for p in poly]
        if geom.polygon_area_2d(poly) > 0.0:
            poly.reverse()
        # find corners compute angle between segments at each point
        angles = []
        for i in range(len(poly)):
            p0, p1, p2 = poly[i - 1], poly[i], poly[(i + 1) % len(poly)]
            angle = geom.angle_between_edges_2d(p0, p1, p2)
            angles.append((angle, i))
        angles.sort()
        corners = [(poly[aa[1]][0], poly[aa[1]][1]) for aa in angles[:4]]

        # find the index of the first corner
        start_idx = 0
        for i, p in enumerate(poly):
            if (p[0], p[1]) in corners:
                start_idx = i
                break
        poly = poly[start_idx:] + poly[:start_idx]
        idx = []
        for i, p in enumerate(poly):
            if (p[0], p[1]) in corners:
                idx.append(i)
        self._coverage_builder = CoverageBuilder(cov_wkt=self._inputs['wkt'], cov_name=self._new_coverage_name)
        idx.append(len(poly))
        poly.append(poly[0])
        poly = [tuple(xyz) for xyz in poly]
        for i in range(1, 5):
            start = idx[i - 1]
            end = idx[i]
            self._coverage_builder.add_arc(poly[start:end + 1])


def calc_arc_angle(arc_pts):
    """Calculates the angle between the arc and straight north.

    Args:
        arc_pts (list): list of the x,y,z of the arc points. We only use the nodes
         (first and last point) to compute the direction
    """
    p0 = arc_pts[0]
    p1 = arc_pts[1]
    dx = p1[0] - p0[0]
    dy = p1[1] - p0[1]
    length = math.sqrt(dx * dx + dy * dy)
    uv_x = dx / length
    uv_y = dy / length
    return math.degrees(math.atan2(uv_y, uv_x) - (math.pi / 2.0))


def two_d_unit_vector_from_points(pt1, pt2):
    """Calculates a unit vector from 2 points.

    Args:
        pt1 (iterable(x, y)): coords of point 1
        pt2 (iterable(x, y)): coords of point 2

    Returns:
        tuple (uv_x, uv_y): unit vector
    """
    dx = pt2[0] - pt1[0]
    dy = pt2[1] - pt1[1]
    length = math.sqrt((dx * dx) + (dy * dy))
    return dx / length, dy / length


def line_to_parametric_values(line):
    """Returns a list of the parametric values of a line from 0.0 to 1.0.

    Args:
        line (list(x, y)): list of coordinates

    Returns:
        (list): parametric values
    """
    lengths = [0.0]
    for i in range(1, len(line)):
        lengths.append(lengths[-1] + xy_distance(line[i - 1], line[i]))
    t_vals = [length / lengths[-1] for length in lengths]
    return t_vals


def point_t_value_on_line(pt, line, line_t_vals):
    """Finds the parametric value of a point on a line.

    Args:
        pt (x, y): coordinates of a point
        line (list(x, y)): list of points
        line_t_vals (list(float)): list of t values for the line

    Returns:
        (float): the t value of the point
    """
    sh_pt = Point(pt[0], pt[1])
    for i in range(1, len(line)):
        lstr = LineString([line[i - 1], line[i]])
        dist = lstr.distance(sh_pt)
        if dist < 5.0e-9:
            lstr2 = LineString([line[i - 1], pt])
            factor = lstr2.length / lstr.length
            t_val_diff = line_t_vals[i] - line_t_vals[i - 1]
            pt_t = line_t_vals[i - 1] + (factor * t_val_diff)
            return pt_t
    raise RuntimeError('Point is not on line.')


def subdivide_line_segment(pt_0, pt_1, n_segments):
    """Subdivides input line defined by pt_0, pt_1 into multiple segments.

    Args:
        pt_0 (iterable): x, y, z
        pt_1 (iterable): x, y, z
        n_segments (int): number of segements

    Returns:
        (list): new coordinates of the line
    """
    dx = pt_1[0] - pt_0[0]
    dy = pt_1[1] - pt_0[1]
    new_line = [pt_0]
    factor = 1.0 / n_segments
    for i in range(1, n_segments):
        x = pt_0[0] + (dx * i * factor)
        y = pt_0[1] + (dy * i * factor)
        new_line.append([x, y, 0.0])
    new_line.append(pt_1)
    return new_line


def get_direction_from_edges(prev_pt, pt, next_pt):
    """Compute an average vector orthogonal to 2 segments defined by 3 points.

    Args:
        prev_pt (Tuple(x,y,z)): coordinates
        pt (Tuple(x,y,z)): coordinates
        next_pt (Tuple(x,y,z)): coordinates

    Returns:
        tuples (x,y): unit vector
    """
    if prev_pt is not None and next_pt is not None:
        v1 = (pt[0] - prev_pt[0], pt[1] - prev_pt[1])
        mag_v1 = math.sqrt(v1[0] ** 2 + v1[1] ** 2)
        ov1 = (v1[1] / mag_v1, -v1[0] / mag_v1)
        v2 = (next_pt[0] - pt[0], next_pt[1] - pt[1])
        mag_v2 = math.sqrt(v2[0] ** 2 + v2[1] ** 2)
        ov2 = (v2[1] / mag_v2, -v2[0] / mag_v2)
        vec_add = ov1[0] + ov2[0], ov1[1] + ov2[1]
        vec_mag = math.sqrt(vec_add[0] ** 2 + vec_add[1] ** 2)
        return vec_add[0] / vec_mag, vec_add[1] / vec_mag, 0.0
    else:
        if prev_pt is not None:
            v1 = (pt[0] - prev_pt[0], pt[1] - prev_pt[1])
            mag_v1 = math.sqrt(v1[0] ** 2 + v1[1] ** 2)
            return v1[1] / mag_v1, -v1[0] / mag_v1, 0.0
        else:
            v2 = (next_pt[0] - pt[0], next_pt[1] - pt[1])
            mag_v2 = math.sqrt(v2[0] ** 2 + v2[1] ** 2)
            return v2[1] / mag_v2, -v2[0] / mag_v2, 0.0
