"""A dialog for displaying feature map lines."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtCore import QUrl
from PySide2.QtWidgets import QVBoxLayout

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.data.map_locator_web_parameters import ParametersObject, ParametersWebPage
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.tool_xms.utils.geom_drawer import clear_geometry, DisplayParameters


class CleanShapesViewerDialog(XmsDlg):
    """Visualize shapes."""
    def __init__(self, parent):
        """Initializes the sediment material list and properties dialog.

        Args:
            parent (:obj:`QWidget`): Parent dialog
        """
        super().__init__(parent, 'xmstool_xms.gui.clean_shapes_viewer_dialog')
        self._widgets = {}

        temp_dir = XmEnv.xms_environ_process_temp_directory()
        self._url_file = os.path.join(temp_dir, f'clean_shapes_viewer_{os.getpid()}.html')
        self._display_parameters = None
        # Add widgets to dialog
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        self.setWindowTitle('Map Viewer')
        self.setMinimumSize(300, 300)
        self._widgets['vert_layout'] = QVBoxLayout()
        self._widgets['plot_view'] = ParametersWebPage(self)
        self._widgets['vert_layout'].addWidget(self._widgets['plot_view'])
        self.setLayout(self._widgets['vert_layout'])

    def set_parameters(self, lat, long, zoom, x_min, x_max, y_min, y_max):
        """Set the parameters as a callback from the javascript web page code."""
        self._display_parameters = DisplayParameters()
        self._display_parameters.lat = lat
        self._display_parameters.long = long
        self._display_parameters.zoom = zoom
        self._display_parameters.x_min = x_min
        self._display_parameters.x_max = x_max
        self._display_parameters.y_min = y_min
        self._display_parameters.y_max = y_max

    def update_viewer(self, viewer, url_file):
        """Draw the levee BC coverage in the map.

        Args:
            viewer (QWebEngineView): viewer to update with
            url_file (string): url file location
        """
        map_name = viewer.get_name()
        self._url_file = url_file
        parameters = ParametersObject("parameters", map_name, self)
        self._widgets['plot_view'].page().add_object(parameters)
        self._widgets['plot_view'].page().runJavaScript(
            f"var bounds = {map_name}.getBounds();"
            "var center = bounds.getCenter(); "
            f"window.set_parameters(center.lat, center.lng, {map_name}.getZoom(), "
            "bounds.getWest(), bounds.getEast(), bounds.getSouth(), bounds.getNorth());")

        self._widgets['plot_view'].setUrl(QUrl.fromLocalFile(self._url_file))
        self._widgets['plot_view'].show()

    def clear(self):
        """Draw the levee BC coverage in the map.

        Args:
            arcs (list): List of LineStrings to draw
            extents (list): List of extents
        """
        clear_geometry(self._url_file)
        self._widgets['plot_view'].setUrl(QUrl.fromLocalFile(self._url_file))
        self._widgets['plot_view'].show()
