"""Script used to run TUFLOWFV."""
# 1. Standard python modules
import argparse
import os
import subprocess

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, XmsEnvironment as XmEnv
from xms.components.bases.run_base import RunBase
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.tuflowfv.feedback.solution_import_runner import import_solution_with_feedback
from xms.tuflowfv.file_io import io_util


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def get_tuflowfv_exe():
    """Get the TUFLOWFV SRH exes from the registry.

    Returns:
        str: Path to the TUFLOWFV executable
    """
    settings = SettingsManager(python_path=False)
    package = 'File_Location_Preferences'
    key = 'TUFLOWFV - TUFLOWFV'
    return os.path.normpath(settings.get_setting(package, key, ''))


class SimRunner(RunBase):
    """Manages running the TUFLOWFV executable and loading solutions."""

    def __init__(self, dummy_mainfile=''):
        """Constructor."""
        super().__init__()

    def get_executables(self, sim, query, filelocation):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.ExecutableCommand`):
                The executable objects to run and the action requests that go with it.
        """
        tuflowfv_exe = get_tuflowfv_exe()
        # Run all TUFLOWFV simulations from shared folder.
        tuflowfv_folder = io_util.move_to_shared_folder(filelocation)
        fvc_filename = os.path.join(tuflowfv_folder, 'runs', f'{sim.name}.fvc')
        if not os.path.isfile(fvc_filename):
            XmEnv.report_error(
                f'Unable to run TUFLOWFV because "{fvc_filename}" was not found. Ensure the simulation has been '
                f'exported.'
            )
            return []
        entry_point = os.path.normpath(
            os.path.join(os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points', 'run_model.py')
        )
        cmd = ExecutableCommand(executable=entry_point, model='TUFLOWFV', executable_order=0, display_name='TUFLOWFV',
                                run_weight=100, progress_script='xml_entry_points/tuflowfv_progress.py',
                                executable_is_script=True)
        cmd.add_commandline_arg(f'-e {tuflowfv_exe}')
        cmd.add_commandline_arg(f'-f {fvc_filename}')
        cmd.add_solution_file(self.get_solution_load_actions(sim, query, filelocation)[0])
        return [cmd]

    def get_solution_load_actions(self, sim, query, filelocation):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            list[ActionRequest]: The executable objects to run and the action requests that go with it.
        """
        tuflowfv_folder = io_util.move_to_shared_folder(filelocation)
        fvc_filename = os.path.join(tuflowfv_folder, 'runs', f'{sim.name}.fvc')
        load_sol = ActionRequest(modality='MODAL', class_name='SimRunner', module_name='xms.tuflowfv.dmi.sim_runner',
                                 method_name='read_solution', parameters={'fvc_file': fvc_filename},
                                 main_file=fvc_filename)
        return [load_sol]

    def read_solution(self, query, params, win_cont):
        """Reads the TUFLOWFV Solution.

        Args:
            query (Query): a Query object to communicate with GMS.
            params (list[dict]): Generic map of parameters
            win_cont (QWidget): Parent window
        """
        import_solution_with_feedback(fvc_filename=params[0]['fvc_file'], query=query, parent=win_cont)
        return [], []


def launch_model():
    """Entry point for launching the TUFLOWFV model executable."""
    parser = argparse.ArgumentParser(description='TUFLOWFV Runner')
    parser.add_argument('-e', '--exe_file', help='Path to the TUFLOWFV executable file', required=True)
    parser.add_argument('-f', '--fvc_file', help='TUFLOWFV control input file', required=True)
    args = parser.parse_args()
    fvc_file = args.fvc_file.strip()
    exe_file = args.exe_file.strip()
    io_util.move_to_shared_folder(os.getcwd())  # This will delete the sim folder created by SMS.
    run_folder = os.path.dirname(fvc_file)
    os.chdir(run_folder)  # Now move to the directory containing the .fvc file.
    child_proc_file = os.path.join(XmEnv.xms_environ_temp_directory(), f'{os.getpid()}_child_proc.txt')

    log_file = XmEnv.xms_environ_stdout_file()
    with open(log_file, 'a') as stdout:
        tuflowfv_process = subprocess.Popen([exe_file, os.path.basename(fvc_file)], stdout=stdout, stderr=stdout)
        with open(child_proc_file, 'w') as f:
            f.write(f'{tuflowfv_process.pid}\n')
        tuflowfv_process.communicate()
