"""Mimics information provided by the ncdump utility."""
# 1. Standard python modules
import traceback

# 2. Third party modules
import netCDF4

# 3. Aquaveo modules
from xms.tuflowfv.file_io import io_util

# 4. Local modules


"""
Adapted from SO solution: https://stackoverflow.com/questions/50114015/no-module-named-ncdump
"""


def _print_ncattr(key, ncfile, stream):
    """Prints the NetCDF file attributes for a given key.

    Args:
        key (str): The NetCDF variable key
        ncfile (netCDF4.Dataset): Open handle to the NetCDF file
        stream (TextIOBase): Stream to write output to
    """
    try:
        stream.write(f'\t\ttype: {repr(ncfile.variables[key].dtype)}\n')
        for ncattr in ncfile.variables[key].ncattrs():
            stream.write(f'\t\t{ncattr}: {repr(ncfile.variables[key].getncattr(ncattr))}\n')
    except KeyError:
        stream.write(f'\t\t{key} does not contain variable attributes\n')


def ncdump(filename, stream):
    """Mimics information provided by the ncdump utility.

    Args:
        filename (str): Absolute path to the NetCDF file
        stream (TextIOBase): Stream to write output to
    """
    try:
        with netCDF4.Dataset(filename, 'r') as ncfile:
            # NetCDF global attributes
            nc_attrs = ncfile.ncattrs()
            stream.write('NetCDF Global Attributes:\n')
            for nc_attr in nc_attrs:
                stream.write(f'\t{nc_attr}: {repr(ncfile.getncattr(nc_attr))}\n')
            nc_dims = [dim for dim in ncfile.dimensions]  # list of nc dimensions
            # Dimension shape information.
            stream.write('NetCDF dimension information:\n')
            for dim in nc_dims:
                stream.write(f'\tName: {dim}\n')
                stream.write(f'\t\tsize: {len(ncfile.dimensions[dim])}\n')
                _print_ncattr(dim, ncfile, stream)
            # Variable information.
            nc_vars = [var for var in ncfile.variables]  # list of nc variables
            stream.write("NetCDF variable information:")
            for var in nc_vars:
                if var not in nc_dims:
                    stream.write(f'\tName: {var}\n')
                    stream.write(f'\t\tdimensions: {ncfile.variables[var].dimensions}\n')
                    stream.write(f'\t\tsize: {ncfile.variables[var].size}\n')
                    _print_ncattr(var, ncfile, stream)
    except Exception as ex:
        stream.write(f'Error(s) retrieving ncdump info from {io_util.logging_filename(filename)}:\n')
        traceback.print_exception(type(ex), ex, ex.__traceback__, file=stream)
