"""Qt dialog for defining output point properties."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtGui import QDoubleValidator

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.number_corrector import NumberCorrector

# 4. Local modules
from xms.tuflowfv.gui import assign_bc_consts as const, gui_util
from xms.tuflowfv.gui.output_points_dialog_ui import Ui_OutputPointsDialog


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class OutputPointsDialog(XmsDlg):
    """A dialog showing the output point attribute dialog."""

    def __init__(self, parent, point_data, export_format):
        """Constructor.

        Args:
            parent (QWidget): The parent Qt dialog
            point_data (xr.Dataset): The output point's attribute dataset
            export_format (str): One of '2dm' and 'Shapefile'. Determines how the coverage will be exported to the
                model. Note that many attributes only carry over if using the GIS format.
        """
        super().__init__(parent, 'xms.tuflowfv.gui.output_points_dialog')
        self.point_data = point_data
        self.export_format = export_format  # Global option that can be edited from any point

        self.number_corrector = NumberCorrector(self)
        self.vert_validator = QDoubleValidator(self)
        self.vert_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        self.vert_validator.setBottom(0.0)
        self.vert_validator.setTop(99999.0)

        self.ui = Ui_OutputPointsDialog()
        self.ui.setupUi(self)
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        self._add_combobox_options()
        self._add_validators()
        self._load_data()

    def _add_combobox_options(self):
        """Add items to combobox widgets common to both arcs and points."""
        gui_util.add_combobox_options(const.BC_CBX_OPTS, self.ui.cbx_export_format)

    def _add_validators(self):
        """Add validators to the float edit fields."""
        self.ui.edt_vert_min.setValidator(self.vert_validator)
        self.ui.edt_vert_min.installEventFilter(self.number_corrector)
        self.ui.edt_vert_max.setValidator(self.vert_validator)
        self.ui.edt_vert_max.installEventFilter(self.number_corrector)

    def _load_data(self):
        """Populate widgets from persistent data."""
        # Read options from saved files
        self.ui.edt_label.setText(self.point_data['label'][0].item())
        self.ui.edt_comment.setText(self.point_data['comment'][0].item())
        self.ui.edt_vert_min.setText(str(self.point_data['vert_min'][0].item()))
        self.ui.edt_vert_max.setText(str(self.point_data['vert_max'][0].item()))
        gui_util.set_combobox_from_data(self.ui.cbx_export_format, self.export_format)

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        # Will reassign a new component id outside the dialog
        self.point_data['label'] = self.ui.edt_label.text()
        self.point_data['comment'] = self.ui.edt_comment.text()
        self.point_data['vert_min'] = float(self.ui.edt_vert_min.text())
        self.point_data['vert_max'] = float(self.ui.edt_vert_max.text())
        self.export_format = self.ui.cbx_export_format.currentText()

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        super().accept()
