"""Filter model for the Z modifications table."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.data import sim_data as smd


COL_MODIFICATION_TYPE = 0
COL_SET_ZPTS = 1
COL_GRID_ZPTS = 2
COL_CSV_FILE = 3
COL_CSV_TYPE = 4
COL_LINE_SHAPE = 5
COL_POINT_SHAPES = 6
COL_TEXT_MODIFICATION_TYPE = 'Type'
COL_TEXT_SET_ZPTS = 'Set Zpts'
COL_TEXT_GRID_ZPTS = 'GRID Zpts'
COL_TEXT_CSV_FILE = 'CSV File'
COL_TEXT_CSV_TYPE = 'CSV Type'
COL_TEXT_LINE_SHAPE = 'ZLine Shapes'
COL_TEXT_POINT_SHAPES = 'ZPoint Shapes'
COLUMN_TEXT = [  # Pretty text for the GUI.
    COL_TEXT_MODIFICATION_TYPE,  # Modification type combobox
    COL_TEXT_SET_ZPTS,  # Constant float edit field
    COL_TEXT_GRID_ZPTS,  # Raster file selector
    COL_TEXT_CSV_FILE,  # CSV file selector
    COL_TEXT_CSV_TYPE,  # CSV file type combobox
    COL_TEXT_LINE_SHAPE,  # Zline shapefile tree item selector
    COL_TEXT_POINT_SHAPES,  # ZPoint shapefile tree item selector
]


class ZModificationsFilterModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (QWidget): The parent object.
        """
        super().__init__(parent)

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (int): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (int): The role.

        Returns:
            The data.
        """
        if role != Qt.DisplayRole:
            return super().headerData(section, orientation, role)

        if orientation == Qt.Horizontal:
            if section >= len(COLUMN_TEXT):
                return ''  # Watch out for the hidden columns
            return COLUMN_TEXT[section]
        else:
            return section + 1

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (QModelIndex): The item's model index

        Returns:
            Qt.ItemFlags: Updated flags for the item
        """
        flags = super().flags(index)
        col = index.column()
        flags |= Qt.ItemIsSelectable
        if col not in [COL_MODIFICATION_TYPE, COL_SET_ZPTS, COL_CSV_TYPE]:
            flags = flags & (~Qt.ItemIsEditable)
        else:
            flags |= Qt.ItemIsEditable

        # Enable columns based on type combobox
        mod_type = int(index.model().index(index.row(), COL_MODIFICATION_TYPE).data())
        if col == COL_MODIFICATION_TYPE:
            flags |= Qt.ItemIsEnabled
        elif col == COL_SET_ZPTS:
            if mod_type == smd.ELEV_TYPE_SET_ZPTS:
                flags |= Qt.ItemIsEnabled
            else:
                flags = flags & (~Qt.ItemIsEnabled)
        elif col == COL_GRID_ZPTS:
            if mod_type == smd.ELEV_TYPE_GRID_ZPTS:
                flags |= Qt.ItemIsEnabled
            else:
                flags = flags & (~Qt.ItemIsEnabled)
        elif col in [COL_CSV_FILE, COL_CSV_TYPE]:
            if mod_type == smd.ELEV_TYPE_CELL_CSV:
                flags |= Qt.ItemIsEnabled
            else:
                flags = flags & (~Qt.ItemIsEnabled)
        elif col in [COL_LINE_SHAPE, COL_POINT_SHAPES]:
            if mod_type == smd.ELEV_TYPE_ZLINE:
                flags |= Qt.ItemIsEnabled
            else:
                flags = flags & (~Qt.ItemIsEnabled)
        return flags
