"""Class to write a WaveWatch3 bounc nml file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO
import logging
import shutil

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.wavewatch3.file_io.namelists_defaults_util import WW3NamelistsDefaultsUtil
from xms.wavewatch3.file_io.spec_list_writer import WW3SpecListWriter


class WW3BouncNmlWriter:
    """Class to write a WaveWatch3 bounc nml file (related to boundaries namelist)."""
    def __init__(self, xms_data):
        """Constructor.

        Args:
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
        """
        self._ss = StringIO()
        self._logger = logging.getLogger('xms.wavewatch3')
        self._bound_process = None
        self._grid_process = None
        self._xms_data = xms_data
        self._defaults = WW3NamelistsDefaultsUtil()

    def _write_bounc_nml_file(self):
        """Writes the bouncnamelist file."""
        file_w_path = "ww3_bounc.nml"
        self._write_bounc_header()
        self._write_bound_namelist()
        self._write_end_comments()
        self._flush(file_w_path)

    def _write_bounc_header(self):
        """Writes the header comments for the bounc namelist file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - ww3_bounc.nml - Boundary input post-processing       !\n'
            '! -------------------------------------------------------------------- !\n'
            '\n\n'
        )

    def _write_bound_namelist(self):
        """Writes the BOUND_NML namelist for the input boundaries to preprocess."""
        self._ss.write(
            "! -------------------------------------------------------------------- !\n"
            "! Define the input boundaries to preprocess via BOUND_NML namelist\n"
            "! Note: When using a rotated pole WW3 grid, the input spectra are\n"
            "! always assumed to be formulated on a standard pole.\n"
            "!\n"
            "! * namelist must be terminated with /\n"
            "! * definitions & defaults:\n"
            "!     BOUND%MODE                 = 'WRITE'            ! ['WRITE'|'READ']\n"
            "!     BOUND%INTERP               = 2                  ! interpolation [1(nearest),2(linear)]\n"
            "!     BOUND%VERBOSE              = 1                  ! [0|1|2]\n"
            "!     BOUND%FILE                 = 'spec.list'        ! input _spec.nc listing file\n"
            "! -------------------------------------------------------------------- !\n"
        )
        attrs = self._xms_data.sim_data_model_control.group('run_control')
        self._ss.write('&BOUND_NML\n')
        # Other defaults are probably OK, just need the spec.list file.
        if attrs.parameter('bound_mode').value != 'WRITE':
            self._ss.write("    BOUND%MODE                 = 'READ'\n")
        if attrs.parameter('bound_interp').value != 2:
            self._ss.write("    BOUND%INTERP               = 1\n")
        if attrs.parameter('bound_verbose').value != 1:
            self._ss.write(f"    BOUND%VERBOSE              = {attrs.parameter('bound_verbose').value}\n")
        self._ss.write("    BOUND%FILE                 = 'spec.list'\n")
        self._write_spec_list_file()
        self._ss.write('/\n\n')

    def _write_spec_list_file(self):
        """Writes out the spec.list bound file."""
        spectral_coverages = self._xms_data.spectral_coverages
        spec_list_writer = WW3SpecListWriter(spectral_coverages)
        spec_list_writer.write()

    def _write_end_comments(self):
        """Writes out the end comments at the end of the file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - end of namelist                                      !\n'
            '! -------------------------------------------------------------------- !\n'
            '\n'
        )

    def _flush(self, file_w_path):
        """Writes the StringIO previously processed to a file.

        Args:
            file_w_path (:obj:`str`):  String of the filename to write to.
        """
        f = open(file_w_path, 'w')
        self._ss.seek(0)
        shutil.copyfileobj(self._ss, f, 100000)
        f.close()

    def write(self):
        """Top-level entry point for the WaveWatch3 boundary input post-processing file writer."""
        self._write_bounc_nml_file()
