#-------------------------------------------------------------------------------
# Name:        gsf2vtk.py
# Purpose:     Creates a VTK legacy or VTK XML ASCII file from a GSF (grid
#              specification file). Only hexahedron are supported in the legacy
#              VTK format, (so no quadtree or voronoi support), and if
#              non-hexahedron are found in the GSF, saves the VTK XML format.
#
# Author:      Michael Kennard
#
# Created:     26/09/2014
# Copyright:   Aquaveo 2014
# Licence:     MODFLOWLIB Software License - Version 1.0 - May 2013
#              Copyright 2006-2013 Aquaveo. All rights reserved.
#              www.aquaveo.com www.modflow.com
#
#              Redistribution and use in source and binary forms, with or
#              without modification, are permitted provided that the following
#              conditions are met:
#
#              Redistributions of source code must retain the above copyright
#              notice, this list of conditions and the following disclaimer.
#              Redistributions in binary form must reproduce the above
#              copyright notice, this list of conditions and the following
#              disclaimer in the documentation and/or other materials provided
#              with the distribution. THIS SOFTWARE IS PROVIDED BY AQUAVEO
#              ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
#              NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
#              FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
#              SHALL AQUAVEO OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
#              INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
#              DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
#              SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
#              OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#              LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#              (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
#              THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
#              SUCH DAMAGE.
#
#              The views and conclusions contained in the software and
#              documentation are those of the authors and should not be
#              interpreted as representing official policies, either expressed
#              or implied, of AQUAVEO.
#
#              Redistribution or use of source or binary forms with a
#              commercial product requires written consent of AQUAVEO.
#-------------------------------------------------------------------------------


import sys
import getopt

#-------------------------------------------------------------------------------
def clockwise (xyz, c):
    """Return true if the points are ordered clockwise looking down.

    c:   list of corner point xyz indices (0-based) forming a polygon.
    xyz: list of all cell corner point xyz locations.
    """
    area2x = 0.0    # 2 times the area
    npoints = len(c)
    for i in range(0, npoints):
        if i != npoints - 1:
            area2x = area2x + (xyz[c[i]][0] * xyz[c[i+1]][1])
            area2x = area2x - (xyz[c[i]][1] * xyz[c[i+1]][0])
        else:
            area2x = area2x + (xyz[c[i]][0] * xyz[c[0]][1])
            area2x = area2x - (xyz[c[i]][1] * xyz[c[0]][0])

    if area2x < 0.0:
        return 1
    return 0
#-------------------------------------------------------------------------------
def format(value):
    """Format the float value as a string.

    value: floating point value to format as a string.
    """
    return "%g" % value
#-------------------------------------------------------------------------------
def format2(value):
    """Format the float value as a string with more precision.

    value: floating point value to format as a string.
    """
    return "%.11g" % value
#-------------------------------------------------------------------------------
def write_vtk_legacy(xyz, cell_point_idxs, layerIds, vtk_file):
    """Write the VTK legacy file format. Only handles hexahedron.

    xyz: list of all cell corner point xyz locations.
    cell_point_idxs: list of corner point xyz indices (0-based) for all cells.
	layerIds: the layer number of each cell
    vtk_file: File object to write to.
    """

    npoints = len(xyz)

    # Write header
    print('# vtk DataFile Version 3.0', file=vtk_file)
    print('vtk_file output', file=vtk_file)
    print('ASCII', file=vtk_file)
    print('DATASET UNSTRUCTURED_GRID', file=vtk_file)
    print('POINTS', npoints, 'double', file=vtk_file)

    # Write points
    for i in range(0,npoints,3):
        for j in range(i, min(i+3, npoints)):
            v = ' '.join([format(x) for x in xyz[j]])
            vtk_file.write(v + ' ')
        vtk_file.write('\n')

    # Compute size
    size = 0
    for c in cell_point_idxs:
        size = size + len(c) + 1

    # Write Cells
    # VTK requires either CCW-bottom-top, or CW-top-bottom.
    ncell = len(cell_point_idxs)
    print('CELLS', ncell, size, file=vtk_file)
    for c in cell_point_idxs:
        npoints = len(c)
        if npoints == 8: # Hexahedron
            temp = get_polyhedron_order(xyz, c)
            print(8, ' '.join(map(str, temp)), file=vtk_file)
        else:
            print("Unhandled cell type", file=vtk_file)
    print('', file=vtk_file) # print a blank line

    # Write cell types
    print('CELL_TYPES', ncell, file=vtk_file)
    for c in cell_point_idxs:
        npoints = len(c)
        if npoints == 8:
            print(12, file=vtk_file)
        else:
            print("Unhandled cell type", file=vtk_file)

	# Write layer IDs
    print('FIELD FieldData 1', file=vtk_file)
    print('XMS:Cell%20Layer 1 ', ncell, " int", file=vtk_file)
    count = 0
    for layer in layerIds:
        print(layer, file=vtk_file)
    print('', file=vtk_file) # print a blank line
	
#-------------------------------------------------------------------------------
def eq_eps(a, b, eps):
    """For float numbers. Returns true if a and b are 'equal' (difference is
    less than a small value based on eps = epsilon).

    a: floating point number
    b: floating point number
    eps: small number (epsilon)
    """
    return (abs(a - b) <= abs((a + b) * eps))
#-------------------------------------------------------------------------------
def get_polyhedron_order(xyz, c):
    """Get a list of points in proper order. For connectivity, the order can be
    either CCW-bottom-top, or CW-top-bottom, but for consistency with VTK and
    ease of use with writing the faces, we always make it CW-top-bottom.

    c: list of corner point indices into xyz list for one cell.
    xyz: locations of cell corner points.
    """
    npoints = len(c)
    npoints_over2 = int(npoints / 2)

    eps = 1e-6 # This is arbitrary and a potential problem
    # Sort second half to match first half
    sorted = c
    for i in range(0, npoints_over2):
        x1 = xyz[sorted[i]][0]
        x2 = xyz[sorted[i+npoints_over2]][0]
        y1 = xyz[sorted[i]][1]
        y2 = xyz[sorted[i+npoints_over2]][1]
        if eq_eps(x1, x2, eps) == 0 or eq_eps(y1, y2, eps) == 0:
            for j in range(npoints_over2 + i + 1, npoints):
                x2 = xyz[sorted[j]][0]
                y2 = xyz[sorted[j]][1]
                if eq_eps(x1, x2, eps) and eq_eps(y1, y2, eps):
                    # swap
                    temp = sorted[npoints_over2 + i]
                    sorted[npoints_over2 + i] = sorted[j]
                    sorted[j] = temp
                    break
    temp = []
    if xyz[c[0]][2] > xyz[sorted[npoints_over2]][2]: # Top then bottom
        if clockwise(xyz, sorted[0:npoints_over2]):
            temp = sorted
        else:
            # Make it CCW-bottom-top
            #temp = sorted[npoints_over2:npoints] + sorted[0:npoints_over2]
            # Make it CW-top-bottom to match typical vtk (not required)
            temp = sorted[0:npoints_over2]
            temp.reverse()
            temp1 = sorted[npoints_over2:npoints]
            temp1.reverse()
            # temp = temp + temp1
            temp.extend(temp1)
    else: # Bottom then top
        if clockwise(xyz, sorted[0:4]):
            # Make it CW-top-bot
            temp = sorted[npoints_over2:npoints] + sorted[0:npoints_over2]
        else:
            #temp = sorted
            temp = sorted # Make CW-top-bot to match typical vtk (not required)
            temp.reverse()
    return temp
#-------------------------------------------------------------------------------
def print_6_per_line(x, count, file):
    """Print 6 items per line.

    x: number to print (int or float)
    count: a counter defined outside this function.
    file: the file to print to.
    """
    if count % 6 == 0:
        if count > 0:
            file.write("\n")
        file.write("          ")
    else:
        file.write(" ")
    file.write(str(x))
    count = count + 1
    return count
#-------------------------------------------------------------------------------
def get_face(f, xyz, cell):
    """Return a list of point indices for the given face f, in order of CCW
    looking out of the cell.

    f: integer index of face. 0 = top, 1 = bottom, anything else is a side.
    cell: list of corner point indices into xyz list for one cell.
    xyz: locations of cell corner points.
    """
    in_order = get_polyhedron_order(xyz, cell)  # CW-top-bottom
    face = []
    npoints = len(cell)
    npoints_over2 = int(npoints / 2)
    if f == 0: # top face
        for i in range(0, -npoints_over2, -1):
            c = i
            if i < 0:
                c = npoints_over2 + i
            face.append(in_order[c])
    elif f == 1: # bottom face
        for i in range(npoints_over2, npoints):
            face.append(in_order[i])
    else:
        # This is designed to match typical VTK
        if f != 2:
            face.append(in_order[npoints_over2-f+2])
            face.append(in_order[npoints-f+2])
            face.append(in_order[npoints-f+1])
            face.append(in_order[npoints_over2-f+1])
        else:
            face.append(in_order[0])
            face.append(in_order[npoints_over2])
            face.append(in_order[npoints-1])
            face.append(in_order[npoints_over2-1])
    return face
#-------------------------------------------------------------------------------
def write_vtk_xml(xyz, cell_point_idxs, layerIds, vtk_file):
    """Write the VTK xml file format.

    xyz: list of all cell corner point xyz locations.
    cell_point_idxs: list of corner point xyz indices (0-based) for all cells.
	layerIds: the layer number of each cell
    vtk_file: File object to write to.
    """
    print("<?xml version=\"1.0\"?>", file=vtk_file)
    print("<VTKFile type=\"UnstructuredGrid\" version=\"0.1\" byte_order="
            "\"LittleEndian\" compressor=\"vtkZLibDataCompressor\">",
             file=vtk_file)
    print("  <UnstructuredGrid>", file=vtk_file)
    vtk_file.write("    <Piece NumberOfPoints=\"" + str(len(xyz)) +
                  "\" NumberOfCells=\"" + str(len(cell_point_idxs)) + "\">\n")

    # Write layer IDs
    print("      <CellData>", file=vtk_file)
    print("        <DataArray type=\"Int32\" Name=\"XMS:Cell Layer\" format=\"ascii\">",
             file=vtk_file)
    count = 0
    for layer in layerIds:
        count = print_6_per_line(layer, count, vtk_file)
    print("", file=vtk_file)
    print("        </DataArray>", file=vtk_file)
    print("      </CellData>", file=vtk_file)

    # Write points
    print("      <Points>", file=vtk_file)
    print("        <DataArray type=\"Float64\" Name=\"Points\""
            " NumberOfComponents=\"3\" format=\"ascii\">", file=vtk_file)
    npoints = len(xyz)
    for i in range(0, npoints, 2):
        vtk_file.write("          ")
        for j in range(i, min(i+2, npoints)):
            v = ' '.join([format2(x) for x in xyz[j]])
            vtk_file.write(v)
            if j == i:
                vtk_file.write(' ')
        vtk_file.write('\n')

    print("        </DataArray>", file=vtk_file)
    print("      </Points>", file=vtk_file)

    # Write Cells
    print("      <Cells>", file=vtk_file)
    print("        <DataArray type=\"Int32\" Name=\"connectivity\""
            " format=\"ascii\">", file=vtk_file)
    # VTK requires either CCW-bottom-top, or CW-top-bottom.
    ncell = len(cell_point_idxs)
    count = 0
    for c in cell_point_idxs:
        npoints = len(c)
        temp = []
        cell = get_polyhedron_order(xyz, c)

        # Print the point indices 6 per line
        for i in cell:
            count = print_6_per_line(i, count, vtk_file)

    print("", file=vtk_file)
    print("        </DataArray>", file=vtk_file)

    #Write cell offsets
    print("        <DataArray type=\"Int32\" Name=\"offsets\""
            " format=\"ascii\">", file=vtk_file)
    dict_num_to_type = {4:10, 8:12, 6:42, 5:14}  # Num points -> Cell type dict
    count = 0
    offset = 0
    for c in cell_point_idxs:
        offset = offset + len(c)
        count = print_6_per_line(offset, count, vtk_file)
    print("", file=vtk_file)
    print("        </DataArray>", file=vtk_file)

    # Write cell types
    print("        <DataArray type=\"UInt8\" Name=\"types\" format=\"ascii\">",
             file=vtk_file)
    count = 0
    faces_needed = 0
    for c in cell_point_idxs:
        npoints = len(c)
        cell_type = 42 # Polyhedron
        if npoints in dict_num_to_type:
            cell_type = dict_num_to_type[npoints]
        else:
            faces_needed = 1
        count = print_6_per_line(cell_type, count, vtk_file)
    print("", file=vtk_file)
    print("        </DataArray>", file=vtk_file)

    if faces_needed:
        # Write cell faces
        print("        <DataArray type=\"Int32\" Name=\"faces\""
        " format=\"ascii\">", file=vtk_file)
        # Build the face array
        face_array = []
        offsets = []
        offset = 0
        for c in cell_point_idxs:
            nFaces = int((len(c) / 2) + 2)
            face_array.append(nFaces)
            for f in range(0, nFaces):
                face = get_face(f, xyz, c)
                face_array.append(len(face))
                # face_array = face_array + face
                face_array.extend(face)
            offset = offset + (len(face_array) - offset)
            offsets.append(offset)

        # Print the face array
        count = 0
        for f in face_array:
            count = print_6_per_line(f, count, vtk_file)
        print("", file=vtk_file)
        print("        </DataArray>", file=vtk_file)

        # Write face offsets
        print("        <DataArray type=\"Int32\" Name=\"faceoffsets\""
            " format=\"ascii\">", file=vtk_file)
        count = 0
        for o in offsets:
            count = print_6_per_line(o, count, vtk_file)
        print("", file=vtk_file)
        print("        </DataArray>", file=vtk_file)

    print("      </Cells>", file=vtk_file)
    print("    </Piece>", file=vtk_file)
    print("  </UnstructuredGrid>", file=vtk_file)
    print("</VTKFile>", file=vtk_file)

#-------------------------------------------------------------------------------
def read_gsf_file(xyz, cell_point_idxs, layerIds, gsf_file):
    """Read the GSF file.

    xyz: list of all cell corner point xyz locations.
    cell_point_idxs: list of corner point xyz indices (0-based) for all cells.
    gsf_file: File object to read from.
    """
    # Skip any comments at the top of the file
    line = gsf_file.readline()
    line = line.strip()
    while line[0] == '#':
        line = gsf_file.readline()

    # Skip item 1 - GridType

    # Read item 2 - NCELL, NLAY, IZ, IC
    ncell, nlay, iz, ic = list(map(int, gsf_file.readline().split()))

    # Read item 3 - NVERT
    npoints = int(gsf_file.readline().strip())

    # Read item 4 - vertex XYZs
    for i in range(0, npoints):
        line = gsf_file.readline()
        xyz.append(list(map(float, line.split())))

    # Read item 5 - Cells
    for line in gsf_file:
        cell_data = line.split()
        layerIds.append(cell_data[4])
        verts = list(map(int, cell_data[6:len(cell_data)]))
        verts[:] = [x - 1 for x in verts]  # subtract 1 to make 0-based
        cell_point_idxs.append(verts)

#-------------------------------------------------------------------------------
def xml_required(cell_point_idxs):
    """Return true if xml format required. Legacy format only handles
    hexahedron.

    cell_point_idxs: list of corner point xyz indices (0-based) for all cells.
    """
    for c in cell_point_idxs:
        if len(c) != 8:
            return 1
    return 0

#-------------------------------------------------------------------------------
def main(argv):
    """Main function."""
    xml_specified = 0
    try:
        gsf = argv[0]
        vtk = argv[1]
        if len(argv) > 2:
            xml_specified = 1
    except:
        print('Reads a GSF (grid specification file) and writes a VTK file.')
        print('')
        print('usage:')
        print('python gsf2vtk.py gsf_file vtk_filePrefix [-x]')
        print('-x: Force VTK XML format. Otherwise XML is only used if needed.')
        sys.exit(2)

    with open(gsf, 'r') as gsf_file:
        xyz = []  # XYZ locations of cell corner points
        cell_point_idxs = []  # 0-based indices of cell corner points
        layerIds = [] # layer numbers of each cell
        read_gsf_file(xyz, cell_point_idxs, layerIds, gsf_file)
        if xml_specified or xml_required(cell_point_idxs):
            vtk_filename = vtk + ".vtu"
            with open(vtk_filename, 'w') as vtk_file:
                write_vtk_xml(xyz, cell_point_idxs, layerIds, vtk_file)
        else:
            vtk_filename = vtk + ".vtk"
            with open(vtk_filename, 'w') as vtk_file:
                write_vtk_legacy(xyz, cell_point_idxs, layerIds, vtk_file)

#-------------------------------------------------------------------------------
if __name__ == '__main__':
    main(sys.argv[1:])
