"""Constituent properties."""

# 1. Standard python modules

# 2. Third party modules
import param

# 3. Aquaveo modules

# 4. Local modules
from adhparam.data_frame_utils import create_default_dataframe


# Define column types for general_constituents DataFrame
GENERAL_CONSTITUENTS_COLUMN_TYPES = {
    'ID': 'int64',
    'NAME': 'str',
    'CONC': 'float'
}

# Create default DataFrame
_default_general_constituents_df = create_default_dataframe(GENERAL_CONSTITUENTS_COLUMN_TYPES)


class ConstituentProperties(param.Parameterized):
    salinity = param.Boolean(
        default=False,
        doc='CN SAL: Salinity baroclinic transport active.',
        precedence=1,
    )
    salinity_id = param.Integer(
        default=1,
        bounds=(1, None),
        doc='CN SAL: Constituent ID number.',
        precedence=1.1,
    )
    reference_concentration = param.Number(
        default=35.0,
        bounds=(0, None),
        doc='CN SAL: Reference concentration (ppt).',
        precedence=2,
    )
    temperature = param.Boolean(
        default=False,
        doc='CN TMP: Temperature baroclinic transport active.',
        precedence=3,
    )
    temperature_id = param.Integer(
        default=1,
        bounds=(1, None),
        doc='CN TMP: Constituent ID number.',
        precedence=3.1,
    )
    reference_temperature = param.Number(
        default=12.0,
        softbounds=(-5.0,40),
        doc='CN TMP: Reference temperature degrees C.',
        precedence=4,
    )
    air_water_heat_transfer = param.Boolean(
        default=False,
        doc='CN TMP: Air/water heat transport active.',
        precedence=5,
    )
    short_wave_radiation_series = param.Integer(
        default=0,
        bounds=(0, None),
        doc='DB RAD: Time series id for short wave radiation (w/m^2).',
        precedence=5.1,
    )
    dew_point_temperature_series = param.Integer(
        default=0,
        bounds=(0, None),
        doc='DB RAD: Time series for dew point temperature in degrees C.',
        precedence=5.2,
    )
    vorticity = param.Boolean(
        default=False,
        doc='CN VOR: Vorticity bendway correction active.',
        precedence=6,
    )
    vorticity_id = param.Integer(
        default=1,
        bounds=(1, None),
        doc='CN VOR: Constituent ID number.',
        precedence=6.1,
    )
    vorticity_normalization = param.Number(
        default=1.0,
        bounds=(0, None),
        doc='CN VOR: Normalization factor.',
        precedence=7,
    )
    vorticity_as_term = param.Number(
        default=4.16667,
        bounds=(0, None),
        doc='CN VOR: As empirical term.',
        precedence=8,
    )
    vorticity_ds_term = param.Number(
        default=0.8,
        bounds=(0, None),
        doc='CN VOR: Ds empirical term.',
        precedence=9,
    )
    general_constituents = param.DataFrame(
        default=_default_general_constituents_df,
        doc='CN CON: ID CONC [! Name: NAME] - General constituents, '
            'ID: ID of the constituent, '
            'CONC: Concentration of the constituent.'
            'NAME: The name of the constituent.',
        precedence=10,
    )

    def __init__(self):
        super(ConstituentProperties, self).__init__()
        self._update_salinity()
        self._update_temperature()
        self._update_vorticity()

    @param.depends('salinity', watch=True)
    def _update_salinity(self):
        self.param.salinity_id.precedence = -1
        self.param.reference_concentration.precedence = -1
        if self.salinity:
            self.param.salinity_id.precedence = 1.1
            self.param.reference_concentration.precedence = 2

    @param.depends('temperature', watch=True)
    def _update_temperature(self):
        self.param.temperature_id.precedence = -1
        self.param.reference_temperature.precedence = -1
        self.param.air_water_heat_transfer.precedence = -1
        self.param.short_wave_radiation_series.precedence = -1
        self.param.dew_point_temperature_series.precedence = -1
        if self.temperature:
            self.param.temperature_id.precedence = 3.1
            self.param.reference_temperature.precedence = 4
            self.param.air_water_heat_transfer.precedence = 5
            self.param.short_wave_radiation_series.precedence = 5.1
            self.param.dew_point_temperature_series.precedence = 5.2

    @param.depends('vorticity', watch=True)
    def _update_vorticity(self):
        self.param.vorticity_id.precedence = -1
        self.param.vorticity_normalization.precedence = -1
        self.param.vorticity_as_term.precedence = -1
        self.param.vorticity_ds_term.precedence = -1
        if self.vorticity:
            self.param.vorticity_id.precedence = 6.1
            self.param.vorticity_normalization.precedence = 7
            self.param.vorticity_as_term.precedence = 8
            self.param.vorticity_ds_term.precedence = 9

    def set_not_required(self, value=False):
        self.salinity = value
        self.temperature = value
        self.air_water_heat_transfer = value
        self.vorticity = value
