"""Sediment constituent properties."""

# 1. Standard python modules

# 2. Third party modules
import param

# 3. Aquaveo modules

# 4. Local modules
from adhparam.data_frame_utils import create_default_dataframe


# Column types for sand DataFrame
SAND_COLUMN_TYPES = {
    'ID': 'int64',
    'NAME': 'str',
    'CONCENTRATION': 'float',
    'GRAIN_DIAMETER': 'float',
    'SPECIFIC_GRAVITY': 'float',
    'POROSITY': 'float'
}

# Column types for clay DataFrame
CLAY_COLUMN_TYPES = {
    'ID': 'int64',
    'NAME': 'str',
    'CONCENTRATION': 'float',
    'GRAIN_DIAMETER': 'float',
    'SPECIFIC_GRAVITY': 'float',
    'POROSITY': 'float',
    'CRITICAL_SHEAR_EROSION': 'float',
    'EROSION_RATE': 'float',
    'CRITICAL_SHEAR_DEPOSITION': 'float',
    'FREE_SETTLING_VELOCITY': 'float'
}

# Create default DataFrames
_default_sand_df = create_default_dataframe(SAND_COLUMN_TYPES)
_default_clay_df = create_default_dataframe(CLAY_COLUMN_TYPES)


class SedimentConstituentProperties(param.Parameterized):
    """
    A class to represent the properties of sediment constituents, specifically sand and clay.

    Attributes:
        sand (pd.DataFrame): The properties of sand constituents.
        clay (pd.DataFrame): The properties of clay constituents.
    """

    # DataFrame parameter for sand constituents
    sand = param.DataFrame(
        default=_default_sand_df,
        doc='CN SND: CARD CARD_1 ID CONCENTRATION GRAIN_DIAMETER SPECIFIC_GRAVITY C_3',
        precedence=1,
    )

    # DataFrame parameter for clay constituents
    clay = param.DataFrame(
        default=_default_clay_df,
        doc='CN CLA: CARD CARD_1 ID CONCENTRATION GRAIN_DIAMETER SPECIFIC_GRAVITY POROSITY '
            'CRITICAL_SHEAR_EROSION EROSION_RATE CRITICAL_SHEAR_DEPOSITION FREE_SETTLING_VELOCITY',
        precedence=2,
    )

    def __init__(self):
        """
        Initializes the SedimentConstituentProperties class by calling the superclass initializer.
        """
        super(SedimentConstituentProperties, self).__init__()
