"""This module provides the GradationSettings class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.setting_group import SettingGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class GradationSettings(SettingGroup):
    """Provides definitions for the gradation setings."""

    def __init__(self, name, version, agency, dev):
        """Initializes the gradation settings values."""
        super().__init__(name, version, agency, dev)

        self.input['Sediment particle diameter'] = Variable('Sediment particle diameter', 'float_list', 0,
                                                            [0.0, 0.0, 0.0, 0.0, 0.0], precision=6,
                                                            limits=(-sys.float_info.max, sys.float_info.max),
                                                            unit_type=['length'], native_unit='ft',
                                                            us_units=self.us_short_length,
                                                            si_units=self.si_short_length)
        self.input['Sediment particle diameter'].selected_us_unit = 'mm'
        self.input['Sediment particle diameter'].selected_si_unit = 'mm'
        percent_units = ['%', 'decimal']
        self.input['Percent passing'] = Variable('Percent passing', 'float_list', 1,
                                                 [0.15, 0.5, 0.85, 0.95, 1.0], precision=4,
                                                 limits=(0.0, 1.0), unit_type=['percent'],
                                                 native_unit='decimal', us_units=[percent_units],
                                                 si_units=[percent_units])
        self.input['Percent passing'].selected_us_unit = '%'
        self.input['Percent passing'].selected_si_unit = '%'

        self.input['Default gradation'] = None

        self.gradation_systems = ['USDA', 'UNIFIED', 'AASHTO',]
        self.input['Gradation system'] = Variable('Gradation classification system', 'list', 2,
                                                  self.gradation_systems, complexity=1)

        # The following commented code allows the user to categorize the gradation into:
        # clay, silt, sand, granular, gravel, cobble, and riprap
        # I'm leaving the code here, as it is possible we may want a more defined gradation system in the future
        # See also code in the gradation_layer.py file

        # self.input['Max cohesive d50 size'] = Variable("Maximum D50 size for a cohesive gradation", 'float', 0.004,
        #                                                [],
        #                                                precision=8, unit_type=['length'], native_unit='ft',
        #                                                us_units=self.us_short_length, si_units=self.si_short_length,
        #                                                complexity=2)
        # self.input['Max cohesive d50 size'].selected_us_unit = 'mm'
        # self.input['Max cohesive d50 size'].selected_si_unit = 'mm'
        # self.input['Max cohesive d50 size'].set_val(0.004, self.app_data)

        # self.input['Max silt d50 size'] = Variable("Maximum D50 size for a silt gradation", 'float', 0.0625, [],
        #                                            precision=6, unit_type=['length'], native_unit='ft',
        #                                            us_units=self.us_mid_length, si_units=self.si_mid_length,
        #                                            complexity=2)
        # self.input['Max silt d50 size'].selected_us_unit = 'mm'
        # self.input['Max silt d50 size'].selected_si_unit = 'mm'
        # self.input['Max silt d50 size'].set_val(0.0625, self.app_data)

        # self.input['Max sand d50 size'] = Variable("Maximum D50 size for a sand gradation", 'float', 2.0, [],
        #                                            precision=2, unit_type=['length'], native_unit='ft',
        #                                            us_units=self.us_mid_length, si_units=self.si_mid_length,
        #                                            complexity=2)
        # self.input['Max sand d50 size'].selected_us_unit = 'mm'
        # self.input['Max sand d50 size'].selected_si_unit = 'mm'
        # self.input['Max sand d50 size'].set_val(2.0, self.app_data)

        # self.input['Max granular d50 size'] = Variable("Maximum D50 size for a granular material", 'float', 4.75, [],
        #                                                precision=2, unit_type=['length'], native_unit='ft',
        #                                                us_units=self.us_mid_length, si_units=self.si_mid_length,
        #                                                complexity=2)
        # self.input['Max granular d50 size'].selected_us_unit = 'mm'
        # self.input['Max granular d50 size'].selected_si_unit = 'mm'
        # self.input['Max granular d50 size'].set_val(4.75, self.app_data)

        # self.input['Max gravel d50 size'] = Variable("Maximum D50 size for a gravel gradation", 'float', 64.0, [],
        #                                              precision=2, unit_type=['length'], native_unit='ft',
        #                                              us_units=self.us_mid_length, si_units=self.si_mid_length,
        #                                              complexity=2)
        # self.input['Max gravel d50 size'].selected_us_unit = 'mm'
        # self.input['Max gravel d50 size'].selected_si_unit = 'mm'
        # self.input['Max gravel d50 size'].set_val(64.0, self.app_data)

        # self.input['Max cobble d50 size'] = Variable("Maximum D50 size for a cobble gradation", 'float', 256.0, [],
        #                                              precision=2, unit_type=['length'], native_unit='ft',
        #                                              us_units=self.us_mid_length, si_units=self.si_mid_length,
        #                                              complexity=2)
        # self.input['Max cobble d50 size'].selected_us_unit = 'mm'
        # self.input['Max cobble d50 size'].selected_si_unit = 'mm'
        # self.input['Max cobble d50 size'].set_val(256.0, self.app_data)

        self.input['Uniformity coefficient tolerance'] = Variable(
            "Uniformity coefficient tolerance", 'float', 0.1, [], complexity=2, precision=2)

        self.input['Min D50 for critical shear method'] = Variable(
            "Minimum D50 size used with Shields number equation to determine critical shear method", 'float',
            0.000656168, [], complexity=2, precision=2)  # 0.2mm

        # self.input['Max riprap d50 size'] = Variable("Maximum D50 size for a cohesive gradation", 'float', 0.0, [],
        # precision=2,
        #                                    unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
        #                                    si_units=self.si_mid_length, complexity=2)

        # self.input['Clay soil plot color'] = Variable("Clay soil plot color", 'color', 'rgb', (165, 42, 42),
        #                                                   complexity=2)  # red brown
        # self.input['Clay soil plot fill color'] = Variable("Clay soil plot fill color", 'color', 'rgb',
        #                                                        (205, 133, 63), complexity=2)  # Tan

        self.input['Silty soil plot color'] = Variable("Silty soil plot color", 'color', 'rgb', (192, 182, 173),
                                                       complexity=2)  # brown
        self.input['Silty soil plot fill color'] = Variable("Silty soil plot fill color", 'color', 'rgb',
                                                            (123, 104, 85), complexity=2)  #

        self.input['Sandy soil plot color'] = Variable("Sandy soil plot color", 'color', 'rgb', (80, 67, 42),
                                                       complexity=2)  # yellow
        self.input['Sandy soil plot fill color'] = Variable("Sandy soil plot fill color", 'color', 'rgb',
                                                            (171, 152, 104), complexity=2)  # khaki

        # self.input['Granular plot color'] = Variable("Granular plot color", 'color', 'rgb', (204, 85, 0),
        #                                              complexity=2)  # orange
        # self.input['Granular plot fill color'] = Variable("Granular plot fill color", 'color', 'rgb', (255, 165, 0),
        #                                                   complexity=2)  #

        self.input['Gravel plot color'] = Variable("Gravel plot color", 'color', 'rgb', (101, 102, 92),
                                                   complexity=2)  # dk slate grey
        self.input['Gravel plot fill color'] = Variable("Gravel plot fill color", 'color', 'rgb', (186, 186, 177),
                                                        complexity=2)  #

        # self.input['Cobble plot color'] = Variable("Cobble plot color", 'color', 'rgb', (22, 0, 40),
        #                                            complexity=2)  # blue or dark grey
        # self.input['Cobble plot fill color'] = Variable("Cobble plot fill color", 'color', 'rgb', (255, 255, 127),
        #                                                 complexity=2)  #

        self.input['Riprap plot color'] = Variable("Riprap plot color", 'color', 'rgb', (101, 102, 92),
                                                   complexity=2)  # grey or blue grey
        self.input['Riprap plot fill color'] = Variable("Riprap plot fill color", 'color', 'rgb', (145, 147, 137),
                                                        complexity=2)  #

    def get_input_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        # Putting this import at the top of the file causes a circular import
        # It uses themes for the plotting, but themes use the settings
        from xms.FhwaVariable.core_data.calculator.table_data import TableData
        need_to_create_default_gradation = True
        if self.input['Default gradation'] is not None:
            self.input['Sediment particle diameter'] = self.input['Default gradation'].get_val(). \
                input['Data input'].get_val().input['Sediment particle diameter']
            self.input['Percent passing'] = self.input['Default gradation'].get_val().input['Data input'].\
                get_val().input['Percent passing']
            need_to_create_default_gradation = False

        if need_to_create_default_gradation:

            gradation = {
                'Sediment particle diameter': self.input['Sediment particle diameter'],
                'Percent passing': self.input['Percent passing']
            }

            size = len(gradation['Percent passing'].get_val())
            self.input['Default gradation'] = Variable(
                'Default gradation', 'table', TableData(None, name='Default gradation',
                                                        plot_names=['Default gradation'],
                                                        model_name=self.model_name,
                                                        input=gradation, min_items=1,
                                                        default_number_of_items=size),)

            self.input['Default gradation'].get_val().set_plot_series_options(
                'Default gradation', related_index=0, index=0, x_axis='Sediment particle diameter',
                y_axis='Percent passing', line_color=(102, 51, 0), linetype='solid', line_width=1.5,
                fill_below_line=False, fill_color=(153, 102, 51), pattern='sand')
            self.input['Default gradation'].get_val().set_plot_log_options('Default gradation', True, False)

        input_vars = copy.deepcopy(self.input)

        if 'Sediment particle diameter' in input_vars:
            input_vars.pop('Sediment particle diameter')
        if 'Percent passing' in input_vars:
            input_vars.pop('Percent passing')

        return input_vars

    # def init_values(self, app_data):
    #     """Initializes the gradation values.

    #     Args:
    #         app_data (CalcData): settings CalcData
    #     """
    #     self.input['Max cohesive d50 size'].set_val(0.004, app_data)
    #     self.input['Max silt d50 size'].set_val(0.0625, app_data)
    #     self.input['Max sand d50 size'].set_val(2.0, app_data)
    #     self.input['Max granular d50 size'].set_val(4.75, app_data)
    #     self.input['Max gravel d50 size'].set_val(64.0, app_data)
    #     self.input['Max cobble d50 size'].set_val(256.0, app_data)

    def get_setting_var(self, name, model=None):
        """Returns the variable of a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            model (string): model name

        Returns:
            if the setting was successfully found
            var (FHWAVariable): Value of the setting
        """
        if name == 'Default gradation':
            return True, self.get_default_gradation()
        for cur_variable in self.input:
            if self.input[cur_variable] and self.input[cur_variable].type in ['class', 'UserArray']:
                found, var = self.input[cur_variable].get_val().get_setting_var(name, model=model)
                if found:
                    return found, var
            else:
                if cur_variable == name or self.input[cur_variable] and self.input[cur_variable].name == name:
                    return True, self.input[cur_variable]
        return False, None

    def get_default_gradation(self):
        """Returns the default gradation.

        Returns:
            default gradation (table): default gradation
        """
        if self.input['Default gradation'] is None:
            self.get_input_group()

        return self.input['Default gradation']
