"""Calculator Class."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class Calculator():
    """Calculator class that will perform the calculations."""

    def __init__(self):
        """Initialize the calculator class.

        Args:
            app_name (str): The name of the application.
            version (float): The version of the application.
            agency (str): The agency that developed the application.
            developed_by (str): The developer of the application.
        """
        # Initialize input_dict, results, warnings, and plot_dict
        self.input_dict = {}
        self.results = {}
        self.warnings = {}
        self.plot_dict = {}

        self.can_compute = False

        self.clear_my_own_results = True

        self.cur_wse = None  # Current water surface elevation (used in some calculations & _set_current_value)

    def get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Returns:
            bool: True if can compute
        """
        self.warnings = {}

        self.can_compute = self._get_can_compute()

        return self.can_compute, self.warnings

    # Replace this method! (but left here for cases where there is no data verification needed)
    def _get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        return True

    def compute_data(self, input_dict: dict = None, plot_dict: dict = None, override_can_compute: bool = False):
        """Compute the data.

        Args:
            input_dict (dict): The input data.

        Returns:
            can_compute (bool): True if the data can be computed.
            computed successfully (bool): True if the data was computed successfully.
            results (dict): The computed data.
            warnings (list): A list of notes, warnings, or errors that occurred during the computation.
            plot_dict (dict): The data to be plotted.
        """
        if input_dict is not None:
            self.input_dict = input_dict
        if hasattr(self, 'clear_results'):
            self.clear_results()
        else:
            self.results = {}
        self.warnings = {}
        if plot_dict is not None:
            self.plot_dict = plot_dict

        # Check if the input_dict is valid and contains the necessary data
        self.get_can_compute()
        if not self.can_compute and not override_can_compute:
            return self.can_compute, False, self.results, self.warnings, self.plot_dict

        # Perform the computation
        result = self._compute_data()

        # Assign the results to the results and plot_dict and log any warnings or errors

        # Change the return to True if the computation was successful
        return self.can_compute, result, self.results, self.warnings, self.plot_dict

    # Replace this method! (but left here for cases where there is data verification needed, but no computation)
    def _compute_data(self):
        """Compute the data.

        Args:

        Returns:
            computed successfully (bool): True if the data was computed successfully.
        """
        # Document the source of the equation (if possible):
        #   Give publication, date, page number and link (if there's a bug, that it can be traced back to the source).
        # Assign new variables from the input_dict in similar names as the equation and document units
        return False

    def get_can_compute_with_subdict(self, input_dict: dict, sub_dict: dict):
        """Compute the data with a sub dictionary.

        Args:
            input_dict (dict): The input data.
            sub_dict (dict): The sub dictionary.

        Returns:
            can_compute (bool): True if the data can be computed.
            computed successfully (bool): True if the data was computed successfully.
            results (dict): The computed data.
            warnings (list): A list of notes, warnings, or errors that occurred during the computation.
            plot_dict (dict): The data to be plotted.
        """
        new_dict = copy.deepcopy(input_dict)
        new_dict['calc_data'] = sub_dict
        self.input_dict = new_dict

        return self.get_can_compute()

    def compute_data_with_subdict(self, input_dict: dict, sub_dict: dict, plot_dict: dict = None):
        """Compute the data with a sub dictionary.

        Args:
            input_dict (dict): The input data.
            sub_dict (dict): The sub dictionary.
            plot_dict (dict): The plot dictionary.

        Returns:
            can_compute (bool): True if the data can be computed.
            computed successfully (bool): True if the data was computed successfully.
            results (dict): The computed data.
            warnings (list): A list of notes, warnings, or errors that occurred during the computation.
            plot_dict (dict): The data to be plotted.
        """
        new_dict = copy.deepcopy(input_dict)
        new_dict['calc_data'] = sub_dict
        self.input_dict = new_dict

        if plot_dict is None:
            plot_dict = self.plot_dict

        return self.compute_data(input_dict=new_dict, plot_dict=plot_dict)

    def get_data(self, name: str, result_if_none=None, prior_keys: list = None, input_dict: dict = None):
        """Get the data.

        Args:
            name (str): The name of the data to get.
            result_if_none (any): The value to return if the data is not found.
            prior_keys (list): The list of priority keys to check before the name.
            input_dict (dict): The input data. If not provided, the class's input_dict will be used.

        Returns:
            result (bool): True if the data was found.
            value: The value of the data.
        """
        if prior_keys is None:
            prior_keys = []
        if input_dict is None:
            input_dict = self.input_dict
        for dict_name in ['calc_data', 'project_settings', 'profile', 'app_settings']:
            if dict_name in input_dict:
                result, value = self._get_data_recursive(input_dict[dict_name], name, prior_keys)
                if result:
                    return result, value
        return False, result_if_none

    def _get_data_recursive(self, data_dict: dict, name: str, prior_keys: list = None):
        """Recursively get the data from the input_dict.

        Args:
            data_dict (dict): The input data.
            name (str): The name of the data to get.

        Returns:
            result (bool): True if the data was found.
            value: The value of the data.
        """
        if prior_keys is None:
            prior_keys = []
        if prior_keys == []:
            if name in data_dict:
                return True, data_dict[name]

        current_dict = data_dict
        for key in prior_keys:
            if key in current_dict and isinstance(current_dict[key], dict):
                current_dict = current_dict[key]
            else:
                # If any key in the chain is missing, break out of the loop
                break
        else:
            # If all prioritized keys exist, check for the name in the final dictionary
            if name in current_dict:
                return True, current_dict[name]

        for key, value in data_dict.items():
            if isinstance(value, dict):
                result, r_value = self._get_data_recursive(value, name, prior_keys)
                if result:
                    return result, r_value
            else:
                if key == name:
                    return True, value
        return False, None

    def _set_current_value(self, value: any, unknown: str = None) -> bool:
        """Returns the low and high value for a given unknown.

        Args:
            value (float): sets the current value of the unknown variable
            unknown (string): unknown variable that we are calculating

        Returns:
            True, if successful; otherwise, False
        """
        if not unknown:
            if 'calc_data' in self.input_dict and 'Calculate' in self.input_dict['calc_data']:
                unknown = self.input_dict['calc_data']['Calculate']
            else:
                return False

        if unknown == 'Head':
            self.cur_wse = value
            return True
        else:
            if 'calc_data' in self.input_dict and unknown in self.input_dict['calc_data']:
                self.input_dict['calc_data'][unknown] = value
                return True
        return False

    def check_float_vars_to_greater_zero(self, var_list: list, result: bool = True):
        """Check if the variables are greater than 0.

        Args:
            var_name (string): variable name
            result (bool): variable value

        Returns:
            True if the variable is greater than 0, False otherwise
        """
        _, zero_tol = self.get_data('Zero tolerance', 1e-6)
        for var_name in var_list:
            data_result, value = self.get_data(var_name)
            if data_result:
                if isinstance(value, float):
                    if value <= zero_tol:
                        self.warnings[var_name] = f'Please enter {var_name}'
                        result = False
                else:
                    self.warnings[var_name] = f'{var_name} not entered correctly'
                    result = False
            else:
                self.warnings[var_name] = f'{var_name} not found'
                result = False

        return result

    def check_int_vars_to_greater_zero(self, var_list: list, result: bool = True):
        """Check if the variables are greater than 0.

        Args:
            var_name (string): variable name
            result (bool): variable value

        Returns:
            True if the variable is greater than 0, False otherwise
        """
        for var_name in var_list:
            data_result, value = self.get_data(var_name)
            if data_result:
                if isinstance(value, int):
                    if value <= 0:
                        self.warnings[var_name] = f'Please enter {var_name}'
                        result = False
                else:
                    self.warnings[var_name] = f'{var_name} not entered correctly'
                    result = False
            else:
                self.warnings[var_name] = f'{var_name} not found'
                result = False

        return result

    def get_plot_item_by_name(self, plot_options: dict, item_name: str):
        """Get the plot item by name.

        Args:
            plot_options (dict): The plot options.
            name (str): The name of the plot item.

        Returns:
            PlotOptions: The plot item.
        """
        for data_series in plot_options['Data series'].values():
            for plot_item in data_series['Plot series'].values():
                if plot_item['Name'] == item_name:
                    return plot_item
        return None

    def get_item_by_indices(self, plot_options: dict, data_series_index: int, series_index: int):
        """Get the item by name.

        Args:
            plot_name (str): The name of the plot item.
            data_series_index (int): index of the data series
            series_index (int): index of the series

        Returns:
            PlotOptions: The plot item.
        """
        if data_series_index not in plot_options['Data series']:
            return None
        if series_index not in plot_options['Data series'][data_series_index]['Plot series']:
            return None

        return plot_options['Data series'][data_series_index]['Plot series'][series_index]

    def set_item_by_name(self, plot_options: dict, plot_name: str):
        """Set the item by name.

        Args:
            plot_options (dict): The plot options.
            plot_name (str): The name of the plot item.
        """
        for data_series in self.input_dict['calc_data']['Plot options'][plot_name]['Data series'].values():
            for plot_item in data_series['Plot series'].values():
                if plot_item['Name'] == plot_options['Name']:
                    plot_item.update(plot_options)
                    break

    def set_item_by_indices(self, plot_name: str, series_index: int, new_item):
        """Set the item by name.

        Args:
            plot_name (str): The name of the plot item.
            data_series_index (int): index of the data series
            series_index (int): index of the series
            new_item (?): item of data to set
        """
        if plot_name not in self.plot_dict:
            return False
        # if data_series_index not in self.input_dict['calc_data']['Plot options'][plot_name]['Data series']:
        #     return False
        # if series_index not in self.input_dict['calc_data']['Plot options'][plot_name]['Data series'][
        #         data_series_index]['Plot series']:
        #     return False

        self.plot_dict[plot_name]['series'][series_index] = new_item
        return True

    def get_plot_subdict_and_key_by_name(self, subdict_name: str, plot_type: str = 'series', plot_name: str = None,
                                         plot_dict: dict = None):
        """Get the plot line by name.

        Args:
            subdict_name (str): The name of the sub dictionary.
            plot_type (str): The type of the plot: 'series', 'lines', 'points'.
            plot_name (str): The name of the plot. If not provided, the 1st plot with the subdict name will be returned
            plot_dict (dict): The plot dictionary. If not provided, the class's plot_dict will be used.

        Returns:
            PlotOptions: The plot line.
            key: The key of the sub dictionary if the name is found, None otherwise.
        """
        if plot_dict is None:
            plot_dict = self.plot_dict

        if plot_name is not None:
            return self._get_plot_subdict_and_key_by_name(subdict_name, plot_name, plot_dict, plot_type)

        for plot_name in plot_dict:
            result, key = self._get_plot_subdict_and_key_by_name(subdict_name, plot_name, plot_dict, plot_type)
            if result is not None:
                return result, key

        return None, None

    @staticmethod
    def _get_plot_subdict_and_key_by_name(subdict_name: str, plot_name: str, plot_dict: dict,
                                          plot_type: str = 'series'):
        """Get the item by name.

        Args:
            subdict_name (str): The name of the sub dictionary.
            plot_name (str): The name of the plot item.
            plot_type (str): The type of the plot: 'series', 'lines', 'points'.
            plot_dict (dict): The plot dictionary. If not provided, the class's plot_dict will be used.

        Returns:
            PlotOptions: The plot item.
            key: The key of the sub dictionary if the name is found, None otherwise.
        """
        if plot_name not in plot_dict:
            return None, None
        if plot_name == plot_type:
            result, key = Calculator._check_for_name_in_plot_dict(plot_dict[plot_name], subdict_name)
            if result is not None or plot_type not in plot_dict[plot_name]:
                return result, key
        if isinstance(plot_dict[plot_name], int) or isinstance(plot_dict[plot_name], float) or \
                plot_type not in plot_dict[plot_name]:
            return None, None
        return Calculator._check_for_name_in_plot_dict(plot_dict[plot_name][plot_type], subdict_name)

    @staticmethod
    def _check_for_name_in_plot_dict(plot_dict: dict, subdict_name: str):
        """Check if the name is in the sub dictionary.

        Args:
            subdict (dict): The sub dictionary.
            name (str): The name to check.

        Returns:
            dict: The sub dictionary if the name is found, None otherwise.
            subdict: the key of the sub dictionary if the name is found, None otherwise.
        """
        for subdict in plot_dict:
            if 'Name' in plot_dict[subdict]:
                if plot_dict[subdict]['Name'] == subdict_name:
                    return plot_dict[subdict], subdict
            if 'Label' in plot_dict[subdict]:
                if plot_dict[subdict]['Label'] == subdict_name:
                    return plot_dict[subdict], subdict
            if 'Labels' in plot_dict[subdict]:
                for label in plot_dict[subdict]['Labels']:
                    if label == subdict_name:
                        return plot_dict[subdict], subdict
        return None, None
