"""SeriesOptions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.calculator.calculator_list import CalcOrVarlist
from xms.FhwaVariable.core_data.calculator.plot.point_options import PointOptions
from xms.FhwaVariable.core_data.calculator.plot.series_options import SeriesOptions
from xms.FhwaVariable.core_data.variables.variable import Variable


class DataSeriesOptions(VariableGroup):
    """Class that defines the data for plotting options."""

    def __init__(self, input_list=None, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Plot Series Options.

        Args:
            input_list (dict): dictionary of input variables
            app_data (AppData): application data (for settings)
            model_name (str): name of the model
            project_uuid (uuid): project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'DataSeriesOptions'
        self.type = 'DataSeriesOptions'

        self.input_list = input_list

        # Input
        self.dataset_names = []
        self.input_names = {}
        if input_list is not None:
            for input in input_list:
                self.dataset_names.append(input_list[input].name)
                self.input_names[input] = input_list[input].name

        # Other arguments (if any) can be added to args and kwargs
        args = ()  # Add positional arguments here if needed
        kwargs = {'input_list': input_list}  # Add other keyword arguments as needed

        self.input['Plot points'] = Variable('Plot points', 'calc_list', CalcOrVarlist(
            PointOptions, app_data=app_data, model_name=model_name, project_uuid=project_uuid,
            show_define_btn=False, default_name='Plot point',
            default_plural_name='Plot points', select_one=False, min_number_of_items=0, initial_num_items=0,
            show_number=False, show_duplicate=False, show_delete=False, args=args, kwargs=kwargs), complexity=1)

        self.input['Plot series'] = Variable('Plot series', 'calc_list', CalcOrVarlist(
            SeriesOptions, app_data=app_data, model_name=model_name, project_uuid=project_uuid,
            show_define_btn=False, default_name='Plot series',
            default_plural_name='Plot series', select_one=False, min_number_of_items=0, initial_num_items=1,
            show_number=False, show_duplicate=False, show_delete=False, args=args, kwargs=kwargs), complexity=1)

        self.warnings = []
        self.results = {}
        # self.results['Results'] = Variable('Results', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

    def set_plot_series_options(self, series_index, x_axis, y_axis, name='', line_color=None, linetype=None,
                                line_width=None, fill_below_line=None, fill_color=None, pattern=None, density=None):
        """Set the plot options.

        Args:
            series_index (int): index of the series
            x_axis (str): name of dataset to use for the y-axis data
            y_axis (str): name of dataset to use for the y-axis data
            name (str): name of the series
            line_color (color tuple): line color
            linetype (str): line type ['solid', 'dashed', 'dotted', 'dash-dot';]
            line_width (float): line width
            fill_below_line (bool): fill below line
            fill_color (color tuple): fill color
            pattern (str): fill pattern
            density (int): fill pattern density
        """
        if series_index == len(self.input['Plot series'].value.item_list):
            self.input['Plot series'].value.add_item_to_list()
        elif series_index >= len(self.input['Plot series'].value.item_list):
            return

        self.input['Plot series'].value.item_list[series_index].set_plot_series_options(
            x_axis=x_axis, y_axis=y_axis, name=name, line_color=line_color, linetype=linetype, line_width=line_width,
            fill_below_line=fill_below_line, fill_color=fill_color, pattern=pattern, density=density)

    def set_plot_point_options(self, index, name='', point_color=None, point_marker=None, marker_size=None):
        """Set the plot options.

        Args:
            index (int): index of the plot
            name (str): name of the plot
            point_color (color tuple): color of the point
            point_marker (str): marker type
            marker_size (float): size of the marker
        """
        if index == len(self.input["Plot points"].value.item_list):
            self.input["Plot points"].value.add_item_to_list()
        elif index >= len(self.input["Plot points"].value.item_list):
            return

        self.input["Plot points"].value.item_list[index].set_plot_point_options(
            name=name, point_color=point_color, point_marker=point_marker, marker_size=marker_size)

    def get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        return True, ''

    def get_input_group(self, unknown=None, app_data=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = copy.deepcopy(self.input)

        return input_vars

    def get_key(self, axis):
        """Get the key for the axis.

        Args:
            axis (str): axis ['X axis', 'Y axis', 'XX axis', 'YY axis']
        """
        dataset_name = self.input[axis].get_val()
        for key in self.input_names:
            if self.input_names[key] == dataset_name:
                return key

    def get_item_by_name(self, item_name):
        """Gets an item by its name.

        Args:
            item_name (str): name of the item to get

        Returns:
            item: the item
        """
        return self.input['Plot series'].value.get_item_by_name(item_name)

    def set_item_by_name(self, new_item):
        """Gets an item by its name.

        Args:
            item_name (str): name of the item to get

        Returns:
            item: the item
        """
        return self.input['Plot series'].value.set_item_by_name(new_item)

    def get_item_by_indices(self, series_index):
        """Gets an item by its name.

        Args:
            data_series_index (int): index of the data series
            series_index (int): index of the series
            new_item (?): item of data to set

        Returns:
            item: the item
        """
        if series_index < len(self.input['Plot series'].value.item_list):
            return self.input['Plot series'].value.item_list[series_index]
        return None

    def set_item_by_indices(self, series_index, new_item):
        """Sets an item by its name.

        Args:
            data_series_index (int): index of the data series
            series_index (int): index of the series
            new_item (?): item of data to set

        Returns:
            item: the item
        """
        if series_index == len(self.input['Plot series'].value.item_list):
            self.input['Plot series'].value.add_item_to_list()
        if series_index < len(self.input['Plot series'].value.item_list):
            self.input['Plot series'].value.item_list[series_index] = copy.deepcopy(new_item)
            return True
        return False
