"""SeriesOptions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class SeriesOptions(VariableGroup):
    """Class that defines the data for plotting options."""

    def __init__(self, input_list=None, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Plot Series Options.

        Args:
            input_list (dict): dictionary of input variables
            app_data (AppData): application data (for settings)
            model_name (str): name of the model
            project_uuid (uuid): project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'SeriesOptions'
        self.type = 'SeriesOptions'

        self.input_list = input_list

        # Input
        self.linetypes = ['solid', 'dashed', 'dotted', 'dash-dot']
        self.patterns = ['solid', 'no fill',
                         # Building materials
                         'concrete', 'brick', 'metal',
                         # Soil materials
                         'clay', 'earth', 'sand', 'rock', 'grass', 'peat', 'bituminous',
                         # Patterns
                         'diagonal', 'reverse diagonal', 'vertical', 'horizontal', 'grid', 'diamond',
                         'small dots', 'medium dots', 'large dots', 'stars']

        self.dataset_names = []
        self.input_names = {}
        if input_list is not None:
            for input in input_list:
                self.dataset_names.append(input_list[input].name)
                self.input_names[input] = input_list[input].name

        self.x_var = None
        self.y_var = None
        self.xx_var = None
        self.yy_var = None

        self.input['Name'] = Variable('Series name', 'string', 'new series')
        self.input['Name yy'] = Variable('Series name yy', 'string', 'new series yy')
        self.input['Display'] = Variable('Display', 'bool', True)
        self.input['X axis'] = Variable('X-Axis', 'list', 0, self.dataset_names, complexity=3)
        self.input['Y axis'] = Variable('Y-Axis', 'list', 0, self.dataset_names, complexity=3)
        self.input['Plot yy'] = Variable('Plot on YY-Axis', 'bool', False, [], complexity=3,
                                         note='Plot on the right axis')
        self.input['XX axis'] = Variable('XX-Axis', 'list', 0, self.dataset_names, complexity=3)
        self.input['YY axis'] = Variable('YY-Axis', 'list', 0, self.dataset_names, complexity=3)
        self.input['Line color'] = Variable('Line color', 'color', 'rgb', (256, 256, 256))
        self.input['Line type'] = Variable('Line type', 'list', 0, self.linetypes)
        self.input['Line width'] = Variable('Line width', 'float', 1.0)
        self.input['Fill below line'] = Variable('Fill below line', 'bool', False)
        self.input['Fill color'] = Variable('Fill color', 'color', 'rgb', (256, 256, 256))
        self.input['Fill pattern'] = Variable('Fill pattern', 'list', 0, self.patterns)
        self.input['Fill pattern density'] = Variable('Fill pattern density', 'int', 1)

        self.input['Line color yy'] = Variable('Line color yy', 'color', 'rgb', (256, 256, 256))
        self.input['Line type yy'] = Variable('Line type yy', 'list', 0, self.linetypes)
        self.input['Line width yy'] = Variable('Line width yy', 'float', 1.0)
        self.input['Fill below line yy'] = Variable('Fill below line yy', 'bool', False)
        self.input['Fill color yy'] = Variable('Fill color yy', 'color', 'rgb', (256, 256, 256))
        self.input['Fill pattern yy'] = Variable('Fill pattern yy', 'list', 0, self.patterns)
        self.input['Fill pattern density yy'] = Variable('Fill pattern density yy', 'int', 1)

        self.warnings = []
        self.results = {}
        # self.results['Results'] = Variable('Results', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

    def set_plot_series_options(self, x_axis, y_axis, name='', line_color=None, linetype=None,
                                line_width=None, fill_below_line=None, fill_color=None, pattern=None, density=None):
        """Set the plot options.

        Args:
            x_axis (str): name of dataset to use for the y-axis data
            y_axis (str): name of dataset to use for the y-axis data
            name (str): name of the series
            line_color (color tuple): line color
            linetype (str): line type ['solid', 'dashed', 'dotted', 'dash-dot';]
            line_width (float): line width
            fill_below_line (bool): fill below line
            fill_color (color tuple): fill color
            pattern (str): fill pattern
            density (int): fill pattern density
        """
        if x_axis not in self.input['X axis'].value_options:
            self.input['X axis'].value_options.append(x_axis)

        if y_axis not in self.input['Y axis'].value_options:
            self.input['Y axis'].value_options.append(y_axis)

        self.input['X axis'].set_val(x_axis)
        self.input['Y axis'].set_val(y_axis)

        if name is not None and name != '':
            self.name = name
            self.input['Name'].set_val(name)
        # if series is not None:
        #     self.input['series index'].set_val(series)
        if line_color is not None:
            self.input['Line color'].set_val(line_color)
        if linetype is not None:
            self.input['Line type'].set_val(linetype)
        if line_width is not None:
            self.input['Line width'].set_val(line_width)
        if fill_below_line is not None:
            self.input['Fill below line'].set_val(fill_below_line)
        if fill_color is not None:
            self.input['Fill color'].set_val(fill_color)
        if pattern is not None:
            self.input['Fill pattern'].set_val(pattern)
        if density is not None:
            self.input['Fill pattern density'].set_val(density)

    def get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        return True, ''

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = copy.deepcopy(self.input)

        y_data = copy.deepcopy(self.dataset_names)
        if input_vars['X axis'].get_val() in y_data:
            y_data.remove(input_vars['X axis'].get_val())
        yy_data = copy.deepcopy(self.dataset_names)
        if input_vars['Y axis'].get_val() in yy_data:
            yy_data.remove(input_vars['Y axis'].get_val())
        if not input_vars['Plot yy'].get_val():
            input_vars.pop('XX axis')
            input_vars.pop('YY axis')

            input_vars.pop('Line color yy')
            input_vars.pop('Line type yy')
            input_vars.pop('Line width yy')
            input_vars.pop('Fill below line yy')
            input_vars.pop('Fill color yy')
            input_vars.pop('Fill pattern yy')
            input_vars.pop('Fill pattern density yy')

        return input_vars

    def get_key(self, axis):
        """Get the key for the axis.

        Args:
            axis (str): axis ['X axis', 'Y axis', 'XX axis', 'YY axis']
        """
        dataset_name = self.input[axis].get_val()
        for key in self.input_names:
            if self.input_names[key] == dataset_name:
                return key
        return dataset_name

    @staticmethod
    def convert_linetype_text_to_symbol(linetype):
        """Convert the line type text to a symbol.

        Args:
            linetype (str): line type text

        Returns:
            str: line type symbol
        """
        if linetype == 'solid':
            return '-'
        elif linetype == 'dashed':
            return '--'
        elif linetype == 'dotted':
            return ':'
        elif linetype == 'dash-dot':
            return '-.'

    @staticmethod
    def convert_pattern_text_to_symbol(pattern, density):
        """Convert the line type text to a symbol.

        Args:
            pattern (str): line type text
            density (int): density of the pattern

        Returns:
            str: pattern symbol
        """
        if pattern == 'solid':
            return ''
        if pattern == 'no fill':
            return 'no fill'
        # Building materials
        if pattern == 'concrete':
            return 'st'
        if pattern == 'brick':
            return 'b'
        if pattern == 'metal':
            return 'm'
        # Soil materials
        if pattern == 'clay':
            return 'c'
        if pattern == 'earth':
            return 'e'
        if pattern == 'sand':
            return 's'
        if pattern == 'rock':
            return 'r'
        if pattern == 'grass':
            return 'g'
        if pattern == 'peat':
            return '~'
        if pattern == 'bituminous':
            return 'u'
        # Patterns
        if pattern == 'diagonal':
            return '/' * density
        if pattern == 'reverse diagonal':
            return '\\' * density
        if pattern == 'vertical':
            return '|' * density
        if pattern == 'horizontal':
            return '-' * density
        if pattern == 'grid':
            return '+' * density
        if pattern == 'diamond':
            return 'x' * density
        if pattern == 'small dots':
            return '.' * density
        if pattern == 'medium dots':
            return 'o' * density
        if pattern == 'large dots':
            return 'O' * density
        if pattern == 'stars':
            return '*' * density

    def get_plot_options_dict(self):
        """Get the plot options dictionary."""
        plot_options = {}

        x_key = self.get_key('X axis')
        y_key = self.get_key('Y axis')
        xx_key = self.get_key('XX axis')
        yy_key = self.get_key('YY axis')

        x_var = self.x_var
        y_var = self.y_var
        xx_var = self.xx_var
        yy_var = self.yy_var
        if x_var is None:
            if x_key in self.input_list:
                x_var = self.input_list[x_key]
            if y_key in self.input_list:
                y_var = self.input_list[y_key]
            if yy_key in self.input_list:
                yy_var = self.input_list[yy_key]
            if xx_key in self.input_list:
                xx_var = self.input_list[xx_key]

        plot_options['Name'] = self.input['Name'].get_val()
        plot_options['Name yy'] = self.input['Name yy'].get_val()
        plot_options['X axis'] = self.input['X axis'].get_val()
        plot_options['Y axis'] = self.input['Y axis'].get_val()
        plot_options['x var'] = copy.deepcopy(x_var)
        plot_options['y var'] = copy.deepcopy(y_var)
        plot_options['xx var'] = copy.deepcopy(xx_var)
        plot_options['yy var'] = copy.deepcopy(yy_var)
        plot_options['Plot yy'] = self.input['Plot yy'].get_val()
        plot_options['XX axis'] = self.input['XX axis'].get_val()
        plot_options['YY axis'] = self.input['YY axis'].get_val()
        plot_options['Line color'] = self.input['Line color'].get_val()
        plot_options['Line type'] = self.convert_linetype_text_to_symbol(
            self.input['Line type'].get_val())
        plot_options['Line width'] = self.input['Line width'].get_val()
        plot_options['Fill below line'] = self.input['Fill below line'].get_val()
        plot_options['Fill color'] = self.input['Fill color'].get_val()
        plot_options['Fill pattern'] = self.convert_pattern_text_to_symbol(
            self.input['Fill pattern'].get_val(), self.input['Fill pattern density'].get_val())

        if plot_options['Plot yy']:
            plot_options['Line color yy'] = self.input['Line color yy'].get_val()
            plot_options['Line type yy'] = self.convert_linetype_text_to_symbol(
                self.input['Line type yy'].get_val())
            plot_options['Line width yy'] = self.input['Line width yy'].get_val()
            plot_options['Fill below line yy'] = self.input['Fill below line yy'].get_val()
            plot_options['Fill color yy'] = self.input['Fill color yy'].get_val()
            plot_options['Fill pattern yy'] = self.convert_pattern_text_to_symbol(
                self.input['Fill pattern yy'].get_val(), self.input['Fill pattern density yy'].get_val())

        return plot_options
