"""Setting Group Class."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import base64
import copy
import json
import os
from pathlib import Path
import platform

# 2. Third party modules
# from PySide6.QtCore import QByteArray
try:
    import winreg
except ImportError:
    winreg = None

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.variable_group import VariableGroup


class SettingGroup(VariableGroup):
    """A base class that defines the basics of a defined group of variables."""
    def __init__(self, app_name: str = None, version: str = None, agency: str = None, developed_by: str = None,
                 app_data: str = None):
        """Initializes the Setting Group.

        Args:
            app_name (string): Application or model name
            version (float): Version of the model
            agency (string): Agency that owns/sponsors the model
            developed_by (string): Organization that developed the model
            app_data (AppData): Application data (for settings)
        """
        if app_name is None or version is None or agency is None or developed_by is None:
            raise ValueError("SettingGroup must be initialized with app_name, version, agency, and developed_by.")

        super().__init__(app_data=app_data, model_name=app_name, project_uuid=None)

        self.set_name_ver_agency_dev(app_name, version, agency, developed_by)

    def set_name_ver_agency_dev(self, app_name: str, version: str, agency: str, developed_by: str):
        """Sets the data (name, version, agency, developed by) that is used for registry keys.

        Args:
            app_name (string): Application or model name
            version (float): Version of the model
            agency (string): Agency that owns/sponsors the model
            developed_by (string): Organization that developed the model
        """
        self.app_name = app_name
        self.version = version
        self.agency = agency
        self.developed_by = developed_by

        self._set_name_ver_agency_dev_recursive(self.input, app_name, version, agency, developed_by)

    def _set_name_ver_agency_dev_recursive(self, input_item, app_name: str, version: str, agency: str,
                                           developed_by: str):
        """Sets the data (name, version, agency, developed by) that is used for registry keys.

        Args:
            input_item (dict): Input item to set the data for
            app_name (string): Application or model name
            version (float): Version of the model
            agency (string): Agency that owns/sponsors the model
            developed_by (string): Organization that developed the model
        """
        if input_item is None:
            return
        for input in input_item:
            if isinstance(input_item[input], dict):
                self._set_name_ver_agency_dev_recursive(input_item[input], app_name, version, agency, developed_by)

            elif input_item[input] and hasattr(input_item[input].get_val(), 'set_name_ver_agency_dev'):
                input_item[input].get_val().set_name_ver_agency_dev(app_name, version, agency, developed_by)

    def get_setting(self, name, return_in_case_of_failure=None, model=None):
        """Returns a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            return_in_case_of_failure (varies): if the setting isn't found, this is returned

        Returns:
            if the setting was successfully found
            value (vaires): Value of the setting or return_in_case_of_failure
        """
        found, var = self.get_setting_var(name, model=model)
        if found:
            return True, var.get_val()
        return False, return_in_case_of_failure

    def set_setting(self, name, new_value, app_data=None, model=None):
        """Changes a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            new_value (varies): new value to set
            app_data (AppSettingsData): Settings CalcData, sometimes needed

        Returns:
            if the setting was successfully found
            value (vaires): Value of the setting or return_in_case_of_failure
        """
        for cur_variable in self.input:
            if isinstance(self.input[cur_variable], dict):
                if cur_variable == model:
                    found, var = self.set_setting_var_list(name, new_value, self.input[cur_variable], app_data,
                                                           model)
                    if found:
                        return found, var
            elif self.input[cur_variable].type in ['class', 'UserArray']:
                found, var = self.input[cur_variable].get_val().set_setting_var(name, new_value, app_data,
                                                                                model=model)
                if found:
                    return found, var
            else:
                if cur_variable == name or self.input[cur_variable].name == name:
                    self.input[cur_variable].set_val(new_value, app_data)
                    return True, self.input[cur_variable]
        return False, None

    def get_setting_var(self, name, model=None):
        """Returns the variable of a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            model (string): model name

        Returns:
            if the setting was successfully found
            var (FHWAVariable): Value of the setting
        """
        found, var = self.get_setting_var_list(name, self.input, model=model)
        if found:
            return found, var
        # input_dual = self.get_input_dual()
        # if model and hasattr(self, 'input_dual') and model in self.input_dual:
        #     found, var = self.get_setting_var_list(name, self.input_dual[model], model=model)
        # elif hasattr(self, 'input_dual'):
        #     for for_model in self.input_dual:
        #         found, var = self.get_setting_var_list(name, self.input_dual[for_model], model=model)
        #         if found:
        #             return found, var
        return found, var

    def get_setting_var_list(self, name, input_list, model=None):
        """Returns the variable of a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting

        Returns:
            if the setting was successfully found
            var (FHWAVariable): Value of the setting
        """
        for cur_variable in input_list:
            if isinstance(input_list[cur_variable], dict):
                if cur_variable == model:
                    found, var = self.get_setting_var_list(name, input_list[cur_variable], model=model)
                    if found:
                        return found, var
                else:
                    found, var = self.get_setting_var_list(name, input_list[cur_variable], model=model)
                    if found:
                        return found, var
            elif input_list[cur_variable].type in ['class', 'UserArray']:
                found, var = input_list[cur_variable].get_val().get_setting_var(name, model=model)
                if found:
                    return found, var
            else:
                if cur_variable == name or input_list[cur_variable].name == name:
                    return True, input_list[cur_variable]
        return False, None

    def set_setting_var(self, name, new_value, app_data=None, model=None):
        """Sets the variable of a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            new_value (FhwaVariable): new value to set
            app_data (AppSettingsData): Settings CalcData, sometimes needed

        Returns:
            if the setting was successfully found
            var (FHWAVariable): Value of the setting
        """
        found, var = self.set_setting_var_list(name, new_value, self.input, app_data, model=model)
        if found:
            return found, var

        return found, var

    def set_setting_var_list(self, name, new_value, input_list, app_data=None, model=None):
        """Sets the variable of a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            new_value (FhwaVariable): new value to set
            app_data (AppSettingsData): Settings CalcData, sometimes needed

        Returns:
            if the setting was successfully found
            var (FHWAVariable): Value of the setting
        """
        for cur_variable in input_list:
            if input_list[cur_variable] is None:
                continue
            if isinstance(input_list[cur_variable], dict):
                if cur_variable == model:
                    found, var = self.set_setting_var(name, input_list[cur_variable], model=model)
                    if found:
                        return found, var
            elif input_list[cur_variable].type in ['class', 'UserArray']:
                found, var = input_list[cur_variable].get_val().set_setting_var(name, new_value, app_data,
                                                                                model=model)
                if found:
                    return found, var
            else:
                if cur_variable == name or input_list[cur_variable].name == name:
                    result = input_list[cur_variable].set_val(new_value, app_data)
                    return result, input_list[cur_variable]
        return False, None

    def get_input_tab_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = copy.deepcopy(self.get_input_group())

        _, support_dual_input = self.get_setting('Dual column input')
        if support_dual_input:
            for app_name in self.model_list:
                input_vars.pop(app_name)

        return {'Input': input_vars}

    # def get_input_dual_tab_group(self, unknown=None):
    #     """Returns the input group for the user interface.

    #     Args:
    #         unknown (string): variable that is unknown

    #     Returns:
    #         input_vars (list of variables): input group for the user interface's input table
    #     """
    #     dual_input_vars = {}

    #     if len(self.model_list) == 1:
    #         app_name = self.model_list[0]
    #         for cur_variable in self.input[app_name]:
    #             # dual_input_vars[cur_variable] = copy.deepcopy(self.input[app_name][cur_variable])
    #             dual_input_vars[cur_variable] = self.input[app_name][cur_variable].get_val().get_input_group()
    #         return dual_input_vars
    #     return None

    # def set_from_calculator(self, CalcData):
    #     """Gather the data and set it from a CalcData.
    #     Because the data is stored as attributes to make sure they persist,
    #     it doesn't feed into the File IO very well. So this puts the data into
    #     a CalcData that will."""
    #     self.name = CalcData.name
    #     self.type = CalcData.type
    #     if uuid in dir(CalcData):
    #         self.uuid = CalcData.uuid
    #     # We do not want to save all data, just the value and units selection
    #     self.set_value_from_calculator(CalcData, True)
    #     self.set_value_from_calculator(CalcData, False)

    # def set_value_from_calculator(self, CalcData, is_input):
    #     """Save the value and units selection from file IO CalcData."""
    #     calc_group = CalcData.input
    #     self_group = self.input
    #     if not is_input:
    #         calc_group = CalcData.results
    #         self_group = self.results

    #     for grp_name in calc_group:
    #         if grp_name in self.input:
    #             if calc_group[grp_name].type in ['class', 'user array']:
    #                 self_group[grp_name].get_val().set_from_calculator(calc_group[grp_name].get_val())
    #             else:
    #                 self_group[grp_name].value = calc_group[grp_name].value
    #                 self_group[grp_name].selected_us_unit = calc_group[grp_name].selected_us_unit
    #                 self_group[grp_name].value = calc_group[grp_name].selected_si_unit

    def _save_data_to_registry(self, name: str, value, path):
        """Save data to the Windows Registry."""
        try:
            key = winreg.CreateKey(winreg.HKEY_CURRENT_USER, path)
            winreg.SetValueEx(key, name, 0, winreg.REG_SZ, str(value))
            winreg.CloseKey(key)
        except Exception as e:
            print(f"Failed to save data to Windows Registry: {e}")

    def _save_data_to_file(self, name: str, value, file_path):
        """Save or update a key-value pair in a JSON file."""
        try:
            data = {}

            # Check if file exists and load existing data
            if os.path.exists(file_path):
                with open(file_path, 'r') as file:
                    try:
                        data = json.load(file)
                    except json.JSONDecodeError:
                        print("File exists but is not a valid JSON. Proceeding with an empty dictionary.")

            # Update or add the key-value pair, making sure to convert QByteArray to Base64
            # if isinstance(value, QByteArray):
            #     value = base64.b64encode(value.data()).decode('utf-8')
            # Is this necessary? Refactored from above code that used Pyside6, but can it be removed?
            if isinstance(value, bytes):
                value = base64.b64encode(value.decode('utf-8'))
            data[name] = value

            # Write updated data back to the file
            with open(file_path, 'w') as file:
                json.dump(data, file, indent=4)

            print(f"Data saved to file: {file_path}")

        except Exception as e:
            print(f"Failed to save data to file: {e}")

    def save_persistant_app_data(self, name: str, value, index: int = None, screen_setup: str = None):
        """Save data to the appropriate location based on the operating system.

        Args:
            name (string): the name of the key for the data
            value (varies): the value of the data to set
            index (int): the index of the data
            screen_setup (string): the screen setup
        """
        key = name
        if index is not None:
            key = f'{name}_{index}'
        path = self._get_persistent_path()

        if screen_setup is not None:
            path = path / screen_setup

        system = platform.system()
        if system == 'Windows':
            self._save_data_to_registry(key, value, path)
        else:
            self._save_data_to_file(key, value, path)

    def save_persistant_dialog_data(self, screen_setup: str, dialog_name: str, current_model: str, name: str, value,
                                    index: int = None):
        """Save data to the appropriate location based on the operating system.

        Args:
            screen_setup (string): the screen setup
            dialog_name (string): the dialog name
            current_model (string): the current model
            name (string): the name of the key for the data
            value (varies): the value of the data to set
            index (int): the index of the data
        """
        if current_model is None or dialog_name is None:
            raise ValueError("Current model and dialog name must be set before saving dialog data.")
        key = name
        if index is not None:
            key = f'{name}_{index}'
        path = Path(self._get_persistent_path())

        path = path / current_model / dialog_name / screen_setup

        system = platform.system()
        if system == 'Windows':
            self._save_data_to_registry(key, value, str(path))
        else:
            self._save_data_to_file(key, value, path)

    def _retrieve_data_from_registry(self, name: str, path: str):
        """Retrieves data from the registry.

        Args:
            name (string): the name of the key for the data
            path (string): the path to the registry key

        Returns:
            value (varies): Value of the data to set
        """
        try:
            key = winreg.OpenKey(winreg.HKEY_CURRENT_USER, path)
            value, _ = winreg.QueryValueEx(key, name)
            winreg.CloseKey(key)
            return value
        except FileNotFoundError:
            pass
        except Exception:
            pass

    def _retrieve_data_from_file(self, name: str, file_path):
        """Retrieve a key-value pair from a JSON file.

        Returns:
            The value or None.
        """
        value = None
        try:
            data = {}

            # Check if the file exists and load existing data
            if os.path.exists(file_path):
                with open(file_path, 'r') as file:
                    try:
                        data = json.load(file)
                    except json.JSONDecodeError:
                        print("File exists but is not a valid JSON. Proceeding with an empty dictionary.")

            # Update or add the key-value pair
            value = data.get(name, None)

        except Exception as e:
            print(f"Failed to retrieve data from file: {e}")
        return value

    def retrieve_persistent_dialog_data(self, screen_setup: str, dialog_name: str, current_model: str, name: str,
                                        index: int = None):
        """Loads data from the appropriate location based on the operating system."""
        value = None
        key = name
        if index is not None:
            key = f'{name}_{index}'
        path = Path(self._get_persistent_path())

        path = path / current_model / dialog_name / screen_setup

        # if screen_index is not None:
        #     path = path / screen_index

        system = platform.system()
        if system == 'Windows':
            value = self._retrieve_data_from_registry(key, str(path))
        else:
            value = self._retrieve_data_from_file(key, path)
        return value

    def retrieve_persistent_app_data(self, name: str, index: int = None, screen_setup: str = None):
        """Loads data from the appropriate location based on the operating system."""
        value = None
        key = name
        if index is not None:
            key = f'{name}_{index}'
        path = self._get_persistent_path()

        if screen_setup is not None:
            path = path / screen_setup

        system = platform.system()
        if system == 'Windows':
            value = self._retrieve_data_from_registry(key, path)
        else:
            value = self._retrieve_data_from_file(key, path)
        return value

    def _get_app_folder_pattern(self):
        """Get the application folder pattern.

        Returns:
            The application folder pattern.
        """
        if self.developed_by is None or self.agency is None or self.app_name is None or self.version is None:
            raise ValueError("App name, version, agency, and developed by must be set before getting the app folder.")
        return Path(f"Aquaveo/{self.developed_by}/{self.agency}/{self.app_name}/{self.version}/")

    def _get_persistent_path(self) -> str:
        """Get the persistent data path.

        Returns:
            The persistent data path.
        """
        system = platform.system()
        if system == 'Windows':
            path = f'Software\\{self._get_app_folder_pattern()}\\Settings\\'  # noqa W605
        else:
            config_dir = Path('~').expanduser() / '.aquaveo'
            config_dir.mkdir(parents=True, exist_ok=True)
            path = config_dir / self._get_app_folder_pattern() / 'config.json'
        return path
