"""Classes to handle the GUI side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class FileManager:
    """Data class to hold current file and to keep a history."""

    def __init__(self, app_data):
        """Initialize the ProjectManager class."""
        self.app_data = app_data

        self.recent_files_list = []
        self.recent_files_uuid_list = []
        self.current_file = None  # Current file being worked on
        self.unsaved_changes = False

        # Add icon for filetype in recent files
        # icons by file type dictionaries

        self.max_recent_files = 10
        self.path = 'Software/Aquaveo/FHWA/HydraulicToolbox/RecentFiles\\'

        self.retrieve_recent_files()

    def set_current_and_recent_file(self, file_path, max_recent_files=None):
        """Adds a file to the recent file list.

        Args:
            file_path (string): The path to the file
            max_recent_files (int): The maximum number of recent files to keep
        """
        if max_recent_files is not None:
            self.max_recent_files = max_recent_files
        self.current_file = file_path
        if file_path in self.recent_files_list:
            remove_index = self.recent_files_list.index(file_path)
            self.recent_files_list.pop(remove_index)
            self.recent_files_uuid_list.pop(remove_index)
        self.recent_files_list.insert(0, file_path)
        self.recent_files_uuid_list.insert(0, uuid.uuid4())
        if len(self.recent_files_list) > self.max_recent_files:
            self.recent_files_list.pop()
            self.recent_files_uuid_list.pop()

        self.store_recent_files()

    def get_file_by_index(self, index):
        """Returns the file by index."""
        return self.recent_files_list[index]

    def get_file_by_uuid(self, uuid):
        """Returns the file by index."""
        index = self.recent_files_uuid_list.index(uuid)
        if index == -1:
            return None
        return self.recent_files_list[index]

    def get_file_uuid_by_index(self, index):
        """Returns the file by index."""
        return self.recent_files_uuid_list[index]

    def get_recent_file_list(self):
        """Returns the recent file list."""
        return self.recent_files_list

    def get_recent_file_dict(self):
        """Returns the recent file list."""
        rev_recent_files_list = self.recent_files_list.copy()
        rev_recent_files_list.reverse()
        rev_recent_files_uuid_list = self.recent_files_uuid_list.copy()
        rev_recent_files_uuid_list.reverse()
        return dict(zip(rev_recent_files_list, rev_recent_files_uuid_list))

    def get_current_file(self):
        """Returns the current file."""
        return self.current_file

    def clear_current_file(self):
        """Returns the current file."""
        return self.current_file

    def store_recent_files(self):
        """Store recent files in the registry."""
        for index, file in enumerate(self.recent_files_list):
            self.app_data.app_settings.save_persistant_app_data("RecentFile", file, index)

    def retrieve_recent_files(self):
        """Retrieve recent files from the registry."""
        for index in range(self.max_recent_files):
            file = self.app_data.app_settings.retrieve_persistent_app_data("RecentFile", index)
            if file is not None:
                self.recent_files_list.append(file)
                self.recent_files_uuid_list.append(uuid.uuid4())
