"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionAngleOrSlopeCalc:
    """Perform angle or slope conversion computations."""
    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Angle or slope
    # ----------------------------------------------------------------------
    # US Units
    degrees = ['Degrees', 'degrees', 'DEGREES', '°']
    # TODO: Add support for degrees minutes seconds
    degrees_minutes_seconds = ['Degrees Minutes Seconds', 'degrees minutes seconds', 'DEGREES MINUTES SECONDS',
                               '° \' \"', '°\'\"', '° \'\"', '°\' \"']
    radians = ['Radians', 'radians', 'RADIANS', 'Rad', 'rad']
    slope_ratio = ['H:1V', 'h:1v']
    slope = ['ft/ft', 'm/m', 'foot/foot', 'meter/meter']
    slope_percent = ['%', '%', 'percent', 'Percent', 'PERCENT', 'percentage', 'Percentage', 'PERCENTAGE']
    decimal = ['Decimal', 'decimal', 'DECIMAL', 'Dec', 'dec']
    # SI Units

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        # No US/SI complimentary unit for angle
        if from_unit in self.degrees or from_unit in self.degrees_minutes_seconds or from_unit in self.radians or \
                from_unit in self.slope_ratio or from_unit in self.slope_percent or from_unit in self.slope or \
                from_unit in self.decimal:
            return True, from_unit
        return False, ''

    # ----------------------------------------------------------------------------------------------------------------------
    # Angle or slope
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert the Angle or slope unit.

        Args:
            from_unit (string): unit that value is currently in
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Units
        if from_unit in self.degrees:
            return self.convert_from_degrees(to_unit, value)
        elif from_unit in self.radians:
            return self.convert_from_radians(to_unit, value)
        elif from_unit in self.slope_ratio:
            return self.convert_from_slope_ratio(to_unit, value)
        elif from_unit in self.slope_percent:
            return self.convert_from_slope_percent(to_unit, value)
        elif from_unit in self.slope or from_unit in self.decimal:
            return self.convert_from_slope(to_unit, value)
        # SI Units

        return False, value

    def convert_from_degrees(self, to_unit, value):
        """Convert from the degrees unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.degrees:
            return True, value
        elif to_unit in self.radians:
            return True, value * 0.0174532925199433
        elif to_unit in self.slope_ratio:
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, 1 / math.tan(value * 0.0174532925199433)
        elif to_unit in self.slope_percent:
            # Percentage = [ Tan ( Degrees ) ] x 100
            return True, math.tan(value * 0.0174532925199433) * 100.0
        elif to_unit in self.slope or to_unit in self.decimal:
            return True, math.tan(value * 0.0174532925199433)
        # SI
        return False, value

    def convert_from_radians(self, to_unit, value):
        """Convert from the radians unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.degrees:
            return True, value / 0.0174532925199433
        elif to_unit in self.radians:
            return True, value
        elif to_unit in self.slope_ratio:
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, 1 / math.tan(value)
        elif to_unit in self.slope_percent:
            # Percentage = [ Tan ( Degrees ) ] x 100
            return True, math.tan(value) * 100.0
        elif to_unit in self.slope or to_unit in self.decimal:
            return True, math.tan(value)
        # SI
        return False, value

    def convert_from_slope_ratio(self, to_unit, value):
        """Convert from the slope ratio unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.degrees:
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, math.atan(1 / value) / 0.0174532925199433
        elif to_unit in self.radians:
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, math.atan(1 / value)
        elif to_unit in self.slope_ratio:
            return True, value
        elif to_unit in self.slope_percent:
            # Percentage = [ Tan ( Degrees ) ] x 100
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, 1 / value * 100.0
        elif to_unit in self.slope or to_unit in self.decimal:
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, 1 / value
        # SI
        return False, value

    def convert_from_slope_percent(self, to_unit, value):
        """Convert from the slope percent unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.degrees:
            return True, math.atan(value / 100) / 0.0174532925199433
        elif to_unit in self.radians:
            return True, math.atan(value / 100)
        elif to_unit in self.slope_ratio:
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, 100.0 / value
        elif to_unit in self.slope_percent:
            # Percentage = [ Tan ( Degrees ) ] x 100
            return True, value
        elif to_unit in self.slope or to_unit in self.decimal:
            return True, value / 100.0
        # SI
        return False, value

    def convert_from_slope(self, to_unit, value):
        """Convert from the slope unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.degrees:
            return True, math.atan(value) / 0.0174532925199433
        elif to_unit in self.radians:
            return True, math.atan(value)
        elif to_unit in self.slope_ratio:
            if value == 0.0:
                # We cannot divide by zero; should give an undefined, but then I force others to handle that
                return True, 0.0
            return True, 1.0 / value
        elif to_unit in self.slope_percent:
            # Percentage = [ Tan ( Degrees ) ] x 100
            return True, value * 100.0
        elif to_unit in self.slope or to_unit in self.decimal:
            return True, value
        # SI
        return False, value
