"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionEnergyCalc:
    """Perform energy unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Energy
    # ----------------------------------------------------------------------
    ft_lb = ['ft-lb', 'foot-pound', 'Foot-pound', 'FOOT-POUND']
    kc = ['kcal', 'kilocalorie', 'Kilocalorie', 'KILOCALORIE']
    j = ['J', 'Joule', 'JOULE']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.ft_lb:
            return True, 'J'
        if from_unit in self.kc:
            return True, from_unit
        return False, ''

    # ----------------------------------------------------------------------
    # Energy
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert energy.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.j:
            return self.convert_from_j(to_unit, value)
        elif from_unit in self.kc:
            return self.convert_from_kc(to_unit, value)
        elif from_unit in self.ft_lb:
            return self.convert_from_ft_lb(to_unit, value)

        return False, value

    def convert_from_j(self, to_unit, value):
        """Convert from the joules unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.j:
            return True, value
        elif to_unit in self.kc:
            return True, value * 0.000239006
        elif to_unit in self.ft_lb:
            return True, value * 0.737562

        else:
            return False, value

    def convert_from_kc(self, to_unit, value):
        """Convert from the kiloCalorie unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.j:
            return True, value * 4184.0
        elif to_unit in self.kc:
            return True, value
        elif to_unit in self.ft_lb:
            return True, value * 3085.96

        else:
            return False, value

    def convert_from_ft_lb(self, to_unit, value):
        """Convert from the ft*lb unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.j:
            return True, value * 1.3558179483
        elif to_unit in self.kc:
            return True, value * 0.000324048
        elif to_unit in self.ft_lb:
            return True, value

        else:
            return False, value
