"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionkCalc():
    """Perform k unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Weir coefficient
    # ----------------------------------------------------------------------
    k_us = ['k US Units', 'k US', 'ft^(1/3)/s', '(ft)^(1/3)/s']
    k_si = ['k SI Units', 'k SI', 'm^(1/3)/s', '(m)^(1/3)/s']

    def __init__(self):
        """Initialize the ConversionCalc Class.
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.k_us:
            if from_unit in self.k_us[:2]:
                return True, 'k SI'
            elif from_unit in self.k_us[2:]:
                return True, 'm^(1/3)/s'
        return False, ''

    # ----------------------------------------------------------------------
    # Weir coefficient
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert weir units.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.k_us:
            return self.convert_from_k_us(to_unit, value)
        elif from_unit in self.k_si:
            return self.convert_from_k_si(to_unit, value)

        return False, value

    def convert_from_k_us(self, to_unit, value):
        """Convert from the k US unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.k_us:
            return True, value
        elif to_unit in self.k_si:
            return True, value / 1.485977123

        else:
            return False, value

    def convert_from_k_si(self, to_unit, value):
        """Convert from the k SI unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.k_us:
            return True, value * 1.485977123
        elif to_unit in self.k_si:
            return True, value

        else:
            return False, value
