"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionSpecificWeightCalc:
    """Perform specific weight unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Specific Weight
    # ----------------------------------------------------------------------
    lb_ft3_sw = ['lb/ft^3', 'lbf/ft^3', 'lbs/ft^3', 'pound/ft^3', 'pounds/ft^3', 'pounds per cubic foot']
    N_m3 = ['N/m^3', 'N/M^3']
    kn_m3 = ['kN/m^3', 'KN/M^3']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.lb_ft3_sw:
            return True, 'N/m^3'
        return False, ''

    # ----------------------------------------------------------------------
    # Specific Weight
    # ----------------------------------------------------------------------

    def convert_units(self, from_unit, to_unit, value):
        """Convert specific weight.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.N_m3:
            return self.convert_from_n_m3(to_unit, value)
        if from_unit in self.kn_m3:
            return self.convert_from_kn_m3(to_unit, value)
        elif from_unit in self.lb_ft3_sw:
            return self.convert_from_lb_ft3_sw(to_unit, value)

        return False, value

    def convert_from_n_m3(self, to_unit, value):
        """Convert from the n/m^3 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.N_m3:
            return True, value
        elif to_unit in self.kn_m3:
            return True, value / 1000.0
        elif to_unit in self.lb_ft3_sw:
            return True, value * 0.0063658804842887

        else:
            return False, value

    def convert_from_kn_m3(self, to_unit, value):
        """Convert from the kN/m^3 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.N_m3:
            return True, value * 1000.0
        elif to_unit in self.kn_m3:
            return True, value
        elif to_unit in self.lb_ft3_sw:
            return True, value * 6.3658804842887

        else:
            return False, value

    def convert_from_lb_ft3_sw(self, to_unit, value):
        """Convert from the lb/ft^3 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.N_m3:
            return True, value * 157.0874606377
        elif to_unit in self.kn_m3:
            return True, value / 6.3658804842887
        elif to_unit in self.lb_ft3_sw:
            return True, value

        else:
            return False, value
