"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionTimeCalc:
    """perform time unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Time
    # ----------------------------------------------------------------------
    second = ['second', 'sec', 's', 'Second', 'Sec', 'S', 'SECOND', 'SEC',
              'seconds', 'Seconds', 'SECONDS']
    minute = ['minute', 'min', 'm', 'Minute', 'Min', 'M', 'MINUTE', 'MIN',
              'minutes', 'Minutes', 'MINUTES']
    hour = ['hour', 'hr', 'Hour', 'HR', 'HOUR', 'hours', 'Hours', 'HOURS']
    day = ['day', 'Day', 'DAY', 'days', 'Days', 'DAYS']
    week = ['week', 'Week', 'WEEK', 'weeks', 'Weeks', 'WEEKS']
    year = ['year', 'yr', 'Year', 'YEAR', 'YR', 'years', 'Years', 'YEARS']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.second or from_unit in self.minute or from_unit in self.hour or \
           from_unit in self.day or from_unit in self.week or from_unit in self.year:
            return True, from_unit
        return False, ''
    # ----------------------------------------------------------------------
    # Time
    # ----------------------------------------------------------------------

    def convert_units(self, from_unit, to_unit, value, per=False):
        """Convert time.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            per (bool): convert time as per (like mph or mps)

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.second:
            return self.convert_from_second(to_unit, value, per)
        elif from_unit in self.minute:
            return self.convert_from_minute(to_unit, value, per)
        elif from_unit in self.hour:
            return self.convert_from_hour(to_unit, value, per)
        elif from_unit in self.day:
            return self.convert_from_day(to_unit, value, per)
        elif from_unit in self.week:
            return self.convert_from_week(to_unit, value, per)
        elif from_unit in self.year:
            return self.convert_from_year(to_unit, value, per)

        return False, value

    def convert_from_second(self, to_unit, value, per=False):
        """Convert from the second unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            per (bool): convert time as per (like mph or mps)

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        found = False
        factor = 1.0
        if to_unit in self.second:
            found = True
        elif to_unit in self.minute:
            found = True
            factor = 1 / 60.0
        elif to_unit in self.hour:
            found = True
            factor = 1 / 3600.0
        elif to_unit in self.day:
            found = True
            factor = 1 / 86400
        elif to_unit in self.week:
            found = True
            factor = 1 / 604800.0
        elif to_unit in self.year:
            found = True
            factor = 1 / 31536000.0

        if found:
            if per:
                return True, value / factor
            else:
                return True, value * factor

        else:
            return False, value

    def convert_from_minute(self, to_unit, value, per=False):
        """Convert from the minute unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            per (bool): convert time as per (like mph or mps)

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        found = False
        factor = 1.0
        if to_unit in self.second:
            found = True
            factor = 60.0
        elif to_unit in self.minute:
            found = True
        elif to_unit in self.hour:
            found = True
            factor = 1.0 / 60.0
        elif to_unit in self.day:
            found = True
            factor = 1.0 / (60.0 * 24.0)
        elif to_unit in self.week:
            found = True
            factor = 1.0 / (60.0 * 24.0 * 7.0)
        elif to_unit in self.year:
            found = True
            factor = 1.0 / (60.0 * 24.0 * 7.0 * 52.1428571429)

        if found:
            if per:
                return True, value / factor
            else:
                return True, value * factor
        else:
            return False, value

    def convert_from_hour(self, to_unit, value, per=False):
        """Convert from the hour unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            per (bool): convert time as per (like mph or mps)

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        found = False
        factor = 1.0
        if to_unit in self.second:
            found = True
            factor = 60.0 * 60.0
        elif to_unit in self.minute:
            found = True
            factor = 60.0
        elif to_unit in self.hour:
            found = True
        elif to_unit in self.day:
            found = True
            factor = 1.0 / 24.0
        elif to_unit in self.week:
            found = True
            factor = 1.0 / (24.0 * 7.0)
        elif to_unit in self.year:
            found = True
            factor = 1.0 / (24.0 * 7.0 * 52.1428571429)

        if found:
            if per:
                return True, value / factor
            else:
                return True, value * factor
        else:
            return False, value

    def convert_from_day(self, to_unit, value, per=False):
        """Convert from the day unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            per (bool): convert time as per (like mph or mps)

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        found = False
        factor = 1.0
        if to_unit in self.second:
            found = True
            factor = 60.0 * 60.0 * 24.0
        elif to_unit in self.minute:
            found = True
            factor = 60.0 * 24.0
        elif to_unit in self.hour:
            found = True
            factor = 24.0
        elif to_unit in self.day:
            found = True
        elif to_unit in self.week:
            found = True
            factor = 1.0 / 7.0
        elif to_unit in self.year:
            found = True
            factor = 1.0 / (7.0 * 52.1428571429)

        if found:
            if per:
                return True, value / factor
            else:
                return True, value * factor
        else:
            return False, value

    def convert_from_week(self, to_unit, value, per=False):
        """Convert from the day unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            per (bool): convert time as per (like mph or mps)

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        found = False
        factor = 1.0
        if to_unit in self.second:
            found = True
            factor = 60.0 * 60.0 * 24.0 * 7.0
        elif to_unit in self.minute:
            found = True
            factor = 60.0 * 24.0 * 7.0
        elif to_unit in self.hour:
            found = True
            factor = 24.0 * 7.0
        elif to_unit in self.day:
            found = True
            factor = 7.0
        elif to_unit in self.week:
            found = True
        elif to_unit in self.year:
            found = True
            factor = 1.0 / 52.1428571429

        if found:
            if per:
                return True, value / factor
            else:
                return True, value * factor
        else:
            return False, value

    def convert_from_year(self, to_unit, value, per=False):
        """Convert from the year unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            per (bool): convert time as per (like mph or mps)

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        found = False
        factor = 1.0
        if to_unit in self.second:
            found = True
            factor = 60.0 * 60.0 * 24.0 * 7.0 * 52.1428571429
        elif to_unit in self.minute:
            found = True
            factor = 60.0 * 24.0 * 7.0 * 52.1428571429
        elif to_unit in self.hour:
            found = True
            factor = 24.0 * 7.0 * 52.1428571429
        elif to_unit in self.day:
            found = True
            factor = 7.0 * 52.1428571429
        elif to_unit in self.week:
            found = True
            factor = 52.1428571429
        elif to_unit in self.year:
            found = True

        if found:
            if per:
                return True, value / factor
            else:
                return True, value * factor
        else:
            return False, value
