"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionUnitFlowCalc:
    """Perform flow unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Flow
    # ----------------------------------------------------------------------
    # US Units
    cfspf = ['cfs/ft', 'ft^3/s/ft', 'cubic feet per second per foot',
             'cubic feet per second per foot (cfspf)', 'cubic feet per second per foot (ft^3/s/ft)']

    # SI Units
    cmspm = ['cms/m', 'm^3/s/m', 'cubic meters per second per meter', 'cubic meters per second per meter (cms/m)',
             'cubic meter per second (ft^3/s)']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.cfspf:
            return True, 'cms/m'
        return False, ''

    # ----------------------------------------------------------------------------------------------------------------------
    # Flow
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert flow.

        Args:
            from_unit (string): unit that the value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Units
        if from_unit in self.cfspf:
            return self.convert_from_cfspf(to_unit, value)
        # SI Units
        elif from_unit in self.cmspm:
            return self.convert_from_cmspm(to_unit, value)

        return False, value

    def convert_from_cfspf(self, to_unit, value):
        """Convert from the cubic feet per year unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfspf:
            return True, value
        # SI
        elif to_unit in self.cmspm:
            return True, value * 0.09290304

        else:
            return False, value

    def convert_from_cmspm(self, to_unit, value):
        """Convert from the cubic feet per second unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.cfspf:
            return True, value * 10.7639104167
        # SI
        elif to_unit in self.cmspm:
            return True, value

        else:
            return False, value
