"""DialogPresenter Class to create the model-view for the dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.presenter.dialog.plot_presenter import PlotPresenter
from xms.FhwaVariable.interface_adapters.presenter.dialog.table_presenter import TablePresenter
from xms.FhwaVariable.interface_adapters.view_model.dialog.dialog import Dialog
from xms.FhwaVariable.interface_adapters.view_model.window_base import WindowBase


class DialogPresenter:
    """A class to create the model-view for the dialog."""
    def __init__(self, app_data, dlg_uuid):
        """Initialize the DialogPresenter Class.

        Args:
            app_data (AppData): The application data.
            dlg_uuid (uuid): The dialog uuid.
        """
        self.app_data = app_data
        self.table_presenter = TablePresenter(app_data)
        self.plot_presenter = PlotPresenter(app_data.get_theme())

        self.vm_dialog = Dialog(dlg_uuid)

        self.app_view = None

    def run_dialog(self, calcdata, app_view, dlg_view, current_model, modal=None, icon=None, parent_uuid=None):
        """Run the dialog.

        Args:
            calcdata (CalcData): The CalcData.
            app_view (ViewBase): The app view.
            dlg_view (ViewBase): The dialog view.
            current_model (str): The current model.
            modal (bool): True if the dialog is modal.
            icon (str): The icon.
            parent_uuid (uuid): The parent uuid.
        """
        self.app_view = app_view

        self.dialog_viewmodel = self.build_viewmodel_dialog(calcdata)
        self.size_dlg(dlg_view, self.dialog_viewmodel, calcdata, current_model=current_model)

        # dialog = CalculatorDialog(dlg_calculator)
        if icon is not None:
            self.dialog_viewmodel.icon = icon
            # icon = dialog.vm_dialog.icon
        # if icon is not None:
        #     dialog.setWindowIcon(QIcon(icon))

        if modal is not None:
            self.dialog_viewmodel.modal = modal
        # dialog.setModal(modal)
        # if dialog is not None:
        #     dialog.setWindowFlag(Qt.WindowMinimizeButtonHint, True)
        #     dialog.setWindowFlag(Qt.WindowMaximizeButtonHint, True)

        # view.create_dialog(self.dialog_viewmodel)

        if parent_uuid is not None:
            self.dialog_viewmodel.parent_uuid = parent_uuid

        self.update_support_windows(self.app_view)

        dlg_view.create_dialog(self.dialog_viewmodel)
        result = dlg_view.run_dialog()

        return result, calcdata

    def update_support_windows(self, app_view):
        """Update the support windows.

        Args:
            app_view (ViewBase): The app view.
        """
        if self.app_data.support_plot_index is not None:
            index = self.app_data.support_plot_index
            if len(self.app_data.support_windows) > index:
                app_view.update_support_window(index, self.app_data.support_windows[index])
        if self.app_data.support_wiki_index is not None and self.app_data.support_wiki_index != \
                self.app_data.support_plot_index:
            index = self.app_data.support_wiki_index
            if len(self.app_data.support_windows) > index:
                app_view.update_support_window(index, self.app_data.support_windows[index])

    def update_dialog(self, calcdata, view, plot_windows=0):
        """Update the dialog.

        Args:
            calcdata (CalcData): The CalcData.
            view (ViewDialogBase): The view.
            plot_windows (int): The number of plot windows to have open
        """
        self.dialog_viewmodel = self.build_viewmodel_dialog(calcdata, plot_windows)
        view.update_dialog(self.dialog_viewmodel)
        self.update_support_windows(self.app_view)

    def display_image(self, image, view):
        """Display an image.

        Args:
            image: The image to display.
            view (ViewBase): The view.
        """
        return view.display_image(image)

    def build_viewmodel_dialog(self, calcdata, plot_windows=0):
        """Build the dialog.

        Args:
            calcdata (CalcData): The CalcData.
            plot_windows (int): The number of plot windows to have open

        Returns:
            Dialog: The view-model dialog (all the data needed to display the dialog).
        """
        calcdata.initialize()  # This is to make sure the number of items in lists are properly set

        sys_complexity = calcdata.get_system_complexity_index()
        self.vm_dialog.sys_complexity = sys_complexity

        _, self.vm_dialog.dialog_columns = calcdata.get_setting('Dialog columns')
        _, self.vm_dialog.copy_headers = calcdata.get_setting('Copy headers of table to clipboard', True)
        self.vm_dialog.calc_type = calcdata.type

        self.table_presenter.theme = self.app_data.get_theme()
        self.plot_presenter.theme = self.app_data.get_theme()

        if hasattr(calcdata, 'check_list_length'):
            calcdata.check_list_length()
        # if calcdata.get_can_compute():
        calcdata.warnings = []
        calcdata.compute_data()

        if calcdata.class_name == calcdata.name or calcdata.class_name == '':
            self.vm_dialog.window_title = calcdata.name
        else:
            self.vm_dialog.window_title = f"{calcdata.class_name}: {calcdata.name}"
        # self.vm_dialog.icon = calcdata.icon

        self.vm_dialog.plot_windows = plot_windows

        # Input
        input_variable_list = calcdata.get_input_tab_group()
        self.vm_dialog.input_table_tabs = self.build_table(True, calcdata, input_variable_list)

        # Dual Input
        user_allow_dual_input = calcdata.get_setting('Dual column input')
        if calcdata.calc_support_dual_input and user_allow_dual_input:
            dual_variable_list = calcdata.get_input_dual_tab_group()
            if dual_variable_list is not None:
                self.vm_dialog.dual_input_table_tabs = self.build_table(True, calcdata, dual_variable_list)
            else:
                self.vm_dialog.dual_input_table_tabs = None
        else:
            self.vm_dialog.dual_input_table_tabs = None
        self.vm_dialog.stack_input_tables = calcdata.stack_input_tables

        # Warnings
        if calcdata.calc_support_warnings:
            self.vm_dialog.warnings_list = calcdata.check_warnings()
            if len(self.vm_dialog.warnings_list) == 0:
                self.vm_dialog.warnings_list.append("Computations completed with no warnings")
        else:
            self.vm_dialog.warnings_list = None

        # Results
        if calcdata.calc_support_results:
            results_variable_list = calcdata.get_results_tab_group()
            self.vm_dialog.results_table_tabs = self.build_table(False, calcdata, results_variable_list)
        else:
            self.vm_dialog.results_table_tabs = None

        # Plots
        if calcdata.calc_support_plot:
            for plot_name in calcdata.plots:
                self.vm_dialog.plots_dict[plot_name] = self.plot_presenter.build_plot(calcdata, plot_name)
            # self.vm_dialog.plots_dict = calcdata.plot_dict
        else:
            self.vm_dialog.plots_dict = None

        # Docs
        self.vm_dialog.reference_pdfs = calcdata.reference_pdfs

        # TODO:Add logic that will set the help url if the user clicks a help button
        self.vm_dialog.default_help_url = calcdata.help_url

        self.vm_dialog.show_wiki = calcdata.show_wiki
        self.vm_dialog.show_plot = calcdata.show_plot

        if self.app_data.support_plot_index is not None:
            self.vm_dialog.external_plot = True
            index = self.app_data.support_plot_index
            if len(self.app_data.support_windows) > index:
                self.app_data.support_windows[self.app_data.support_plot_index].plots_dict = self.vm_dialog.plots_dict

        if self.app_data.support_wiki_index is not None:
            self.vm_dialog.external_wiki = True
            index = self.app_data.support_wiki_index
            if len(self.app_data.support_windows) > index:
                self.app_data.support_windows[self.app_data.support_wiki_index].wiki_url = \
                    self.vm_dialog.default_help_url

        self.vm_dialog.undo_list = calcdata.command_manager._undo_stack
        self.vm_dialog.redo_list = calcdata.command_manager._redo_stack

        return self.vm_dialog

    def build_table(self, table_is_input, calcdata, variable_list):
        """Build the table.

        Args:
            table_is_input (bool): True if the table is an input table.
            calcdata (CalcData): The CalcData.
            variable_list (list): The list of variables.
        """
        table_tabs = {}
        vertical_tables = calcdata.get_tables_with_vertical_direction()
        if calcdata.read_only:
            table_is_input = False
        for tab_name in variable_list:
            table_is_vertical = tab_name in vertical_tables
            table = self.table_presenter.setup_table(tab_name=tab_name, table_is_input=table_is_input,
                                                     calcdata=calcdata, variable_list=variable_list[tab_name],
                                                     table_is_vertical=table_is_vertical)
            table_tabs[tab_name] = table
        return table_tabs

    def convert_string_to_bool(self, string):
        """Convert a string to a boolean.

        Args:
            string (str): The string to convert.

        Returns:
            bool: The converted boolean value.
        """
        if string.lower() == 'true':
            return True
        elif string.lower() == 'false':
            return False
        else:
            try:
                return bool(int(string))
            except ValueError:
                return None

    def retrieve_size_data(self, window, app_settings, name, current_model: str, screen_setup_str: str,
                           screen_res_x: int, screen_res_y: int):
        """Retrieve the size data.

        Args:
            window (Dialog): The window.
            app_settings (AppSettingsData): The app settings.
            name (str): The name of the window.
            current_model (str): The current model.
            screen_setup_str (str): The screen setup string.
            screen_res_x (int): The screen resolution x.
            screen_res_y (int): The screen resolution y.
        """
        x = None
        y = None
        width = 1200
        height = 800
        maximized = False

        retrieved_x = app_settings.retrieve_persistent_dialog_data(screen_setup_str, name, current_model, 'x')
        retrieved_y = app_settings.retrieve_persistent_dialog_data(screen_setup_str, name, current_model, 'y')
        retrieved_width = app_settings.retrieve_persistent_dialog_data(screen_setup_str, name, current_model,
                                                                       'width')
        retrieved_height = app_settings.retrieve_persistent_dialog_data(screen_setup_str, name, current_model,
                                                                        'height')

        if retrieved_width:
            width = int(retrieved_width)
        if retrieved_height:
            height = int(retrieved_height)
            # Get the resolution of the screen we are running on
        if retrieved_x:
            x = int(retrieved_x)
        if retrieved_y:
            y = int(retrieved_y)

        if screen_res_x < width:
            width = screen_res_x
        if screen_res_y < height:
            height = screen_res_y

        if x is None or y is None:
            x = (screen_res_x - width) / 2
            y = (screen_res_y - height) / 2

        if x + width / 2 > screen_res_x:
            x = screen_res_x - width / 2
        if y + height / 2 > screen_res_y:
            y = screen_res_y - height / 2

        window.x = x
        window.y = y
        window.width = width
        window.height = height

        r_maximized = app_settings.retrieve_persistent_dialog_data(screen_setup_str, name, current_model,
                                                                   'maximized')
        if r_maximized:
            r_maximized = self.convert_string_to_bool(r_maximized)
            if r_maximized is not None:
                maximized = r_maximized
        window.maximized = maximized
        window.minimized = False

    def size_dlg(self, view, dialog: Dialog, calcdata, current_model: str):
        """Size the dialog.

        Starts with a defined size, checks if the calcdata specifies a size, then checks the registry, and
        finally checks the screen size.

        Args:
            view (ViewBase): The view to size.
            dialog (view_model.Dialog): The view model of the dialog to size.
            calcdata (CalcData): The CalcData.
            current_model (str): The current model.
        """
        app_data = calcdata.app_data
        dlg_name = calcdata.name
        if hasattr(calcdata, 'class_name') and calcdata.class_name != '':
            dlg_name = calcdata.class_name
        if dlg_name == 'SettingsCalc' and current_model == '' or current_model is None:
            current_model = 'AppData'

        show_wiki = None
        show_plot = None

        # Get Screen configuration
        if view is not None:
            screens_info = view.get_screens_info()
        screen_setup_str = screens_info['screen_setup_str']
        screen_res_x, screen_res_y = view.get_screen_resolution()

        # Get the previous size from the registry
        app_settings = app_data.app_settings

        self.retrieve_size_data(dialog, app_settings, dlg_name, current_model, screen_setup_str,
                                screen_res_x, screen_res_y)

        retrieved_show_wiki = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name, current_model,
                                                                           'show_wiki')
        retrieved_show_plot = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name, current_model,
                                                                           'show_plot')

        if retrieved_show_wiki:
            retrieved_show_wiki = self.convert_string_to_bool(retrieved_show_wiki)
            if retrieved_show_wiki is not None:
                show_wiki = retrieved_show_wiki
        if retrieved_show_plot:
            retrieved_show_plot = self.convert_string_to_bool(retrieved_show_plot)
            if retrieved_show_plot is not None:
                show_plot = retrieved_show_plot

        # cursor_pos = QCursor.pos()
        # screen = QGuiApplication.screenAt(cursor_pos)
        # screen_geometry = screen.geometry()
        # self.screen_width = screen_geometry.width()
        # self.screen_height = screen_geometry.height()
        # screen_width, screen_height = view.get_screen_width_and_height()
        # dpi = view.get_screen_dpi()
        # scaling_factor = view.get_screen_scaling_factor()

        if show_wiki is None or show_plot is None:
            min_y_size_for_plot_and_wiki = 1440
            if screen_res_y < min_y_size_for_plot_and_wiki:
                show_wiki = False
                show_plot = False
            else:
                show_wiki = True
                show_plot = True

        dialog.show_wiki = show_wiki
        dialog.show_plot = show_plot

        retrieved_splitter_h = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name, current_model,
                                                                            'splitter_h_size')
        if retrieved_splitter_h:
            for index in range(int(retrieved_splitter_h)):
                retrieved_size_x = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name,
                                                                                current_model, 'splitter_h_x', index)
                retrieved_size_y = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name,
                                                                                current_model, 'splitter_h_y', index)
                sizes = []
                if retrieved_size_x:
                    sizes.append(int(retrieved_size_x))
                if retrieved_size_y:
                    sizes.append(int(retrieved_size_y))

                if len(sizes) > 0:
                    dialog.splitter_h.append(sizes)

        retrieved_splitter_v = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name, current_model,
                                                                            'splitter_v_size')
        if retrieved_splitter_v:
            for index in range(int(retrieved_splitter_v)):
                retrieved_size_x = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name,
                                                                                current_model, 'splitter_v_x', index)
                retrieved_size_y = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name,
                                                                                current_model, 'splitter_v_y', index)
                sizes = []
                if retrieved_size_x:
                    sizes.append(int(retrieved_size_x))
                if retrieved_size_y:
                    sizes.append(int(retrieved_size_y))

                if len(sizes) > 0:
                    dialog.splitter_v.append(sizes)

        # Plot Windows
        retrieved_plot_windows = app_settings.retrieve_persistent_dialog_data(screen_setup_str, dlg_name, current_model,
                                                                              'plot_windows')
        if retrieved_plot_windows:
            dialog.plot_windows = int(retrieved_plot_windows)

        # Check if we have saved plot window data (that may be used, if we create new plot windows)
        index = 1
        found = True
        while found:
            plot_name = f'{dlg_name} plot {index}'
            r_x = app_settings.retrieve_persistent_dialog_data(screen_setup_str, plot_name, current_model, 'x')
            if r_x:
                plot_size = WindowBase()
                self.retrieve_size_data(plot_size, app_settings, plot_name, current_model, screen_setup_str,
                                        screen_res_x, screen_res_y)
                dialog.plot_windows_data_list.append(plot_size)
                index += 1
            else:
                found = False

        # Make sure we have a pos/size for new plot windows
        while dialog.plot_windows > len(dialog.plot_windows_data_list):
            plot_size = WindowBase()
            self.retrieve_size_data(plot_size, app_settings, plot_name, current_model, screen_setup_str,
                                    screen_res_x, screen_res_y)
            dialog.plot_windows_data_list.append(plot_size)

        # Turn on the Sizing Gripper
        # TODO: Get Sizing Gripper to show!
        # self.setSizeGripEnabled(True)

        # # Set Splitter sizes
        # retrieved_num_h_splitters = app_settings.retrieve_persistent_dialog_data(dialog.window_title, current_model,
        #                                   'num_h_splitters')
        # retrieved_num_v_splitters = app_settings.retrieve_persistent_dialog_data(dialog.window_title, current_model,
        #                                   'num_v_splitters')
        # # Make sure we match (that columns or other things were not changed)
        # if retrieved_num_h_splitters and retrieved_num_v_splitters:
        #     num_h_splitters = int(retrieved_num_h_splitters)
        #     num_v_splitters = int(retrieved_num_v_splitters)
        #     if num_h_splitters == len(self.splitters_h) and num_v_splitters == len(self.splitters_v):
        #         for index in range(num_h_splitters):
        #             retrieved_state = app_settings.retrieve_persistent_dialog_data(dialog.window_title, current_model,
        #                                       f'HSplitter{index}')
        #             if retrieved_state:
        #                 self.splitters_h[index].restoreState(retrieved_state.encode("utf-8"))

        #             # retrieved_numSizes = settings_calc.retrieve_persistent_dialog_data(dialog.window_title,
        #                                           current_model, f'HSplitter{index}-num')
        #             # if retrieved_numSizes:
        #             #     sizes = []
        #             #     num_sizes = int(retrieved_numSizes)
        #             #     for inner_index in range(num_sizes):
        #             #         retrieved_size = self.CalcData.retrieve_persistent_dialog_data(dialog.window_title,
        #                                           current_model, f'HSplitter{index}-{inner_index}')
        #             #         if retrieved_size:
        #             #             sizes.append(int(retrieved_size))
        #             #     self.splitters_h[index].setSizes(sizes)

        #         for index in range(num_v_splitters):
        #             retrieved_state = app_settings.retrieve_persistent_dialog_data(dialog.window_title, current_model,
        #                                       f'VSplitter{index}')
        #             if retrieved_state:
        #                 self.splitters_v[index].restoreState(retrieved_state.encode("utf-8"))

        #             # retrieved_numSizes = self.CalcData.retrieve_persistent_dialog_data(dialog.window_title,
        #                                                   current_model, f'VSplitter{index}-num')
        #             # if retrieved_numSizes:
        #             #     sizes = []
        #             #     num_sizes = int(retrieved_numSizes)
        #             #     for inner_index in range(num_sizes):
        #             #         retrieved_size = self.CalcData.retrieve_persistent_dialog_data(dialog.window_title,
        #                                               current_model, f'VSplitter{index}-{inner_index}')
        #             #         if retrieved_size:
        #             #             sizes.append(int(retrieved_size))
        #             #     self.splitters_v[index].setSizes(sizes)
